package com.xebialabs.deployit.core.sql.lock

import java.sql.{Connection, ResultSet}

import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import org.springframework.jdbc.core.{ConnectionCallback, JdbcTemplate}

import scala.jdk.CollectionConverters._

trait SqlLockRepository extends LockTableQueries {
  val jdbcTemplate: JdbcTemplate
  implicit val schemaInfo: SchemaInfo

  def executeWithLock(lockName: String, retryCount: Int = 3, retryDelayMillis: Int = 100)(executeInLock: => Unit): Unit = {
    jdbcTemplate.execute(new ConnectionCallback[Unit] {
      override def doInConnection(con: Connection): Unit = {
        var lock: Option[String] = None
        var count: Int = 0
        do {
          if (count > 0) {
            Thread.sleep(retryDelayMillis)
          }
          lock = jdbcTemplate.query(
            schemaInfo.sqlDialect.lockSelectBuilder().where(cond.equals(LockTableSchema.lockColumnName, lockName)).query,
            (rs: ResultSet, _: Int) => { rs.getString(LockTableSchema.lockColumnName.name) }, lockName).asScala.headOption
          jdbcTemplate.update(UPDATE_LOCK, lockName, lockName)
          count += 1
        } while (lock.isEmpty && retryCount >= count)
        lock.getOrElse(throw new IllegalStateException(s"Could not obtain lock '$lockName' after $retryCount retries"))
        executeInLock
      }
    })
  }

  def insertLock(lockName: String): Unit = {
    jdbcTemplate.update(INSERT_LOCK, lockName)
  }

  def deleteLock(lockName: String): Unit = {
    jdbcTemplate.update(DELETE_LOCK, lockName)
  }

}

object LockTableSchema {
  val lockTableName: TableName = TableName("XLD_LOCK_TABLE")

  val lockColumnName: ColumnName = ColumnName("LOCK_NAME")
}

trait LockTableQueries extends Queries {
  import LockTableSchema._
  lazy val UPDATE_LOCK: String = sqlb"UPDATE $lockTableName SET $lockColumnName = ? WHERE $lockColumnName = ?"
  lazy val INSERT_LOCK: String = sqlb"INSERT INTO $lockTableName ($lockColumnName) VALUES  (?)"
  lazy val DELETE_LOCK: String = sqlb"DELETE FROM $lockTableName  where $lockColumnName = ?"
}

object LockNameRegistry {

  val XLD_WORKERS_LOCK: String = "XLD_WORKERS_LOCK"
  val XLD_PROFILES_LOCK: String = "XLD_PROFILES_LOCK"

}
