package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, TableName}
import com.xebialabs.deployit.repository.ExtendedRepositoryMetadataService
import com.xebialabs.deployit.repository.sql.SqlRepositoryMetadataService.repositoryIdName
import com.xebialabs.license.LicenseRepositoryIdException
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Component
@Transactional("mainTransactionManager")
class SqlRepositoryMetadataService (@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                   (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ExtendedRepositoryMetadataService with MetadataQueries {

  // Lazy initialised to prevent frequent querying
  protected var storedRepoId: Option[String] = None

  override def validateAndStoreRepositoryId(repositoryId: String): Unit = {
    (Option(repositoryId), getRepositoryId) match {
      case (Some(newRepoId), Some(current)) => if (current != newRepoId) throw new LicenseRepositoryIdException(repositoryId)
      case (Some(_), None) => saveRepositoryId(repositoryId)
      case _ => // Don't store
    }
  }

  override def getRepositoryId: Option[String] = {
    storedRepoId match {
      case Some(_) =>
      case None => storedRepoId = queryRepositoryId()
    }
    storedRepoId
  }

  private[repository] def queryRepositoryId(): Option[String] =
    jdbcTemplate.queryForList(SELECT, classOf[String], repositoryIdName).asScala.headOption

  override def saveRepositoryId(repositoryId: String): Unit =
    jdbcTemplate.update(INSERT, repositoryIdName, repositoryId)
}

object SqlRepositoryMetadataService {
  final val repositoryIdName = "repository_id"
}

object MetadataSchema {
  val tableName: TableName = TableName("XL_METADATA")

  val name: ColumnName = ColumnName("name")
  val value: ColumnName = ColumnName("value")
}

trait MetadataQueries extends Queries {

  import MetadataSchema._

  val SELECT = sqlb"select $value from $tableName where $name = ?"
  val INSERT = sqlb"insert into $tableName ($name, $value) values (?, ?)"
}
