package ai.digital.deploy.sql.http.enricher

import java.util.function.{BiConsumer, Supplier}

import com.xebialabs.deployit.engine.api.dto.Paging
import grizzled.slf4j.Logging
import org.jboss.resteasy.spi.{HttpResponse, ResteasyProviderFactory}
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Service

@Service
class PaginationService extends Logging {

  @Value("${deploy.server.rest.api.maxPageSize: 1000}")
  private var maxPageSize: Integer = _

  def getLimitedPaging(input: Paging): Paging = { // Limit the pagination size
    if (input.resultsPerPage <= 0 || input.resultsPerPage > maxPageSize) return new Paging(input.page, maxPageSize)
    input
  }

  def addPagingHeaderIfNeeded(setHeader: BiConsumer[String, AnyRef], countTotalResults: Supplier[Integer], paging: Paging): Unit = {
    val limited = getLimitedPaging(paging)
    addPagingHeaderIfNeeded(paging, limited, setHeader, countTotalResults)
  }

  def addPagingHeaderIfNeeded(paging: Paging, limited: Paging, setHeader: BiConsumer[String, AnyRef], countTotalResults: Supplier[Integer]): Unit = {
    if (limited.resultsPerPage > 0) {
      val count = countTotalResults.get
      if (paging.resultsPerPage <= 0 && count > limited.resultsPerPage)
        logger.warn(s"Limiting total number of results $count to maximum ${limited.resultsPerPage}")
      setHeader.accept(Paging.X_TOTAL_COUNT_HEADER, count)
    }
  }

  def toSetHeader(response: HttpResponse): BiConsumer[String, AnyRef] = {
    if (ResteasyProviderFactory.getInstance.getContextData(classOf[HttpResponse]) == null)
      new IgnoreNoHttpResponseContext

    (key: String, value: AnyRef) => response.getOutputHeaders.putSingle(key, value)
  }

  private class IgnoreNoHttpResponseContext extends BiConsumer[String, AnyRef] {
    override def accept(s: String, o: AnyRef): Unit = {
    }
  }

  def setMaxPageSize(maxPageSize: Integer): Unit = {
    this.maxPageSize = maxPageSize
  }
}
