package com.xebialabs.deployit.core.sql

case class TableName(name: String, quoted: Boolean = true) {
  def getIdentifier(identifier: String)(implicit schemaInfo: SchemaInfo): String =
    if (quoted) schemaInfo.sqlDialect.quote(identifier) else identifier


  def build(alias: Option[String])(implicit schemaInfo: SchemaInfo): String = {
    val tableName = getIdentifier(name)
    schemaInfo.schemaName match {
      case Some(schema) =>
        val schemaName = getIdentifier(schema)
        alias.map(schemaInfo.sqlDialect.aliasTableWithSchemaName(tableName, schemaName, _))
          .getOrElse(schemaInfo.sqlDialect.tableWithSchemaName(tableName, schemaName))
      case None =>
        alias.map(schemaInfo.sqlDialect.aliasTable(tableName, _)).getOrElse(tableName)
    }
  }
}

sealed trait Selectable {
  def build(tableAlias: Option[String] = None)(implicit schemaInfo: SchemaInfo): String
}

case class ColumnName(name: String, quoted: Boolean = true) extends Selectable {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = {
    val result = if (quoted) schemaInfo.sqlDialect.quote(name) else name
    tableAlias.map(a => s"$a.$result").getOrElse(result)
  }
  def tableAlias(alias: String): AliasedTableColumnName = AliasedTableColumnName(alias)(this)
  def columnAlias(alias: String): AliasedColumnName = AliasedColumnName(alias)(this)
}

case class AliasedTableColumnName(tableAlias: String)(wrapped: ColumnName) extends Selectable {
  override def build(ignored: Option[String])(implicit schemaInfo: SchemaInfo): String = wrapped.build(Some(tableAlias))
  def columnAlias(alias: String): AliasedColumnName = AliasedColumnName(alias)(this)
}

case class AliasedColumnName(columnAlias: String)(wrapped: Selectable) extends Selectable {
  override def build(ignored: Option[String])(implicit schemaInfo: SchemaInfo): String =
    wrapped match {
      case AliasedTableColumnName(tableAlias) => s"${wrapped.build(Some(tableAlias))} as $columnAlias"
      case otherWrapped => s"${otherWrapped.build(None)} as $columnAlias"
    }
}

abstract class SqlFunction extends Selectable

class SimpleSqlFunction(private val name: String)(private val column: Selectable) extends SqlFunction {
  def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = name + '(' + column.build(tableAlias) + ')'
}

class StaticSqlFunction(private val format: String)(private val args: Any*) extends SqlFunction {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = format.format(args.map {
    case s:Selectable => s.build(tableAlias)
    case o => o
  }: _*)
}

class SqlLiteral(private val value: String) extends Selectable {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = value
}

object SqlFunction {
  val average = new SimpleSqlFunction("avg")(_)
  val countAll = new SqlLiteral("count(*)")
  val countOne = new SqlLiteral("count(1)")
  val lower = new SimpleSqlFunction("lower")(_)
  val sum = new SimpleSqlFunction("sum")(_)
  val max = new SimpleSqlFunction("max")(_)
  val ? = new SqlLiteral("?")
}
