package com.xebialabs.deployit.core

import java.lang.Long
import java.sql.{PreparedStatement, Timestamp, Types}
import java.time.{LocalDateTime, ZoneOffset}
import java.util.{Calendar, GregorianCalendar, TimeZone}
import org.joda.time.{DateTime, DateTimeZone}
package object sql {

  def toTimestamp(dateTime: DateTime) = new Timestamp(dateTime.getMillis)
  def toCalendar(timestamp: Timestamp): Calendar = {
    val calendar = new GregorianCalendar(TimeZone.getTimeZone("UTC"))
    calendar.setTimeInMillis(timestamp.getTime)
    calendar
  }

  def toCalendar(dateTime: DateTime): Calendar = if (dateTime == null) null
    else dateTime.withZone(DateTimeZone.UTC).toGregorianCalendar

  def toDateTime(timestamp: Timestamp) = new DateTime(timestamp.getTime, DateTimeZone.UTC)

  def toDateTime(dateTime: LocalDateTime) = new DateTime(dateTime.atOffset(ZoneOffset.UTC).toInstant.toEpochMilli, DateTimeZone.UTC)

  def mapDateTime(date: AnyRef) = date match {
    case t: Timestamp => toDateTime(t)
    case t: LocalDateTime => toDateTime(t)
    case _ => throw new IllegalArgumentException("Cannot map date of " + date + " to DateTime")
  }

  val UTC_CALENDAR: Calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"))

  def setTimestamp(preparedStatement: PreparedStatement, index: Int, dateTime: DateTime): Unit =
    if (dateTime == null)
      preparedStatement.setNull(index, Types.TIMESTAMP)
    else
      preparedStatement.setTimestamp(index, toTimestamp(dateTime), toCalendar(dateTime))

  def asInteger(any: Any): Integer = any match {
    case i : Integer => i
    case n : Number => n.intValue()
    case _ => throw new IllegalArgumentException(s"Cannot convert $any to Integer.") // should not happen
  }

  def asLong(any: Any): Long = any match {
    case i: Long => i
    case n: Number => n.longValue()
    case _ => throw new IllegalArgumentException(s"Cannot convert $any to Long.") // should not happen
  }

  def orderByClause(orderBy: Seq[OrderBy], tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = mkString(orderBy.map(_.build(tableAlias)), "", " order by ", ", ", "")

  def mkString(iterable: Iterable[_], empty: String, start: String, sep: String, end: String): String =
    if (iterable.isEmpty) empty else iterable.mkString(start, sep, end)

}
