package com.xebialabs.deployit.core.sql

import org.joda.time.DateTime

trait SqlCondition {
  def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String
  def arguments: Seq[Any]
}

object SqlCondition {
  def and(conditions: Seq[SqlCondition]) = AndCondition(conditions)
  def or(conditions: Seq[SqlCondition]) = OrCondition(conditions)
  def equals(column1: Selectable, column2: Selectable) = ColumnsEqualCondition(column1, column2)
  def equals(column: Selectable, value: String) = StringEqualsCondition(column, value)
  def equals(column: Selectable, value: Number) = NumberEqualsCondition(column, value)
  def equals(column: Selectable, value: Boolean) = BooleanEqualsCondition(column, value)
  def equalsNull(column: Selectable) = IsNullCondition(column)
  def equalsNotNull(column: Selectable) = IsNotNullCondition(column)
  def largerThen(column: Selectable, value: Number) = NumberLargerThenCondition(column, value)
  def smallerThen(column: Selectable, value: Number) = NumberSmallerThenCondition(column, value)
  def smallerOrEqual(column: Selectable, value: Number) = NumberSmallerOrEqualCondition(column, value)
  def before(column: Selectable, value: DateTime) = BeforeDateCondition(column, value)
  def after(column: Selectable, value: DateTime) = AfterDateCondition(column, value)
  def between(column: Selectable, startDate: DateTime, endDate: DateTime): SqlCondition = and(Seq(after(column, startDate), before(column, endDate)))
  def in(column: Selectable, values: Iterable[_]) = InCondition(column, values.toSeq)
  def subselect(column: Selectable, subBuilder: QueryBuilder) = SubselectCondition(column, subBuilder)
  def subselectNot(column: Selectable, subBuilder: QueryBuilder) = SubselectNotCondition(column, subBuilder)
  def joinSubselect(condition: SqlCondition) = JoinSubselectCondition(condition)
  def like(column: Selectable, value: String) = LikeCondition(column, value)
  def likeEscaped(column: Selectable, value: String) = LikeEscapedCondition(column, value)
  def not(condition: SqlCondition) = NotCondition(condition)
}

trait ConditionOnColumn extends SqlCondition {
  def column: Selectable
}

case class AndCondition(conditions: Seq[SqlCondition]) extends SqlCondition {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = conditions.map {_.build(tableAlias)}.mkString("(", " and ", ")")
  override def arguments: Seq[Any] = conditions.flatMap{ _.arguments }
}

case class OrCondition(conditions: Seq[SqlCondition]) extends SqlCondition {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = conditions.map {_.build(tableAlias)}.mkString("(", " or ", ")")
  override def arguments: Seq[Any] = conditions.flatMap{ _.arguments }
}

case class BeforeDateCondition(override val column: Selectable, date: DateTime) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} < ?"
  override def arguments = Seq(date)
}

case class AfterDateCondition(override val column: Selectable, date: DateTime) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} >= ?"
  override def arguments = Seq(date)
}

case class ColumnsEqualCondition(column1: Selectable, column2: Selectable) extends SqlCondition {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column1.build(tableAlias)} = ${column2.build(tableAlias)}"
  override def arguments: Seq[Any] = Seq()
}

case class StringEqualsCondition(override val column: Selectable, string: String) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} = ?"
  override def arguments = Seq(string)
}

case class NumberEqualsCondition(override val column: Selectable, number: Number) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} = ?"
  override def arguments = Seq(number)
}

case class BooleanEqualsCondition(override val column: Selectable, boolean: Boolean) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} = ?"
  override def arguments = Seq(boolean)
}

case class IsNullCondition(override val column: Selectable) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} is null"
  override def arguments = Seq()
}

case class IsNotNullCondition(override val column: Selectable) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} is not null"
  override def arguments = Seq()
}

case class NumberLargerThenCondition(column: Selectable, number: Number) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} > ?"
  override def arguments: Seq[Any] = Seq(number)
}

case class NumberSmallerThenCondition(column: Selectable, number: Number) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} < ?"
  override def arguments: Seq[Any] = Seq(number)
}

case class NumberSmallerOrEqualCondition(column: Selectable, number: Number) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} <= ?"
  override def arguments: Seq[Any] = Seq(number)
}

case class InCondition(column: Selectable, values: Seq[_]) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} in " + List.fill(values.size)("?").mkString("(",",",")")
  override def arguments: Seq[Any] = values
}

case class SubselectCondition(override val column: Selectable, query: QueryBuilder) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} in (${query.query})"
  override def arguments: Seq[Any] = query.parameters
}

case class SubselectNotCondition(override val column: Selectable, query: QueryBuilder) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} not in (${query.query})"
  override def arguments: Seq[Any] = query.parameters
}

case class JoinSubselectCondition(condition: SqlCondition) extends SqlCondition {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = condition.build(tableAlias)
  override def arguments: Seq[Any] = condition.arguments
}

case class LikeCondition(override val column: Selectable, value: String) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo) = s"${column.build(tableAlias)} like ?"
  override def arguments = Seq(value)
}

case class LikeEscapedCondition(override val column: Selectable, value: String) extends SqlCondition with ConditionOnColumn {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo) =
    s"${column.build(tableAlias)} like ? ${schemaInfo.sqlDialect.escapeBackSlash()}"
  override def arguments = Seq(value)
}

case class NotCondition(condition: SqlCondition) extends SqlCondition {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"not ${condition.build(tableAlias)}"
  override def arguments: Seq[Any] = condition.arguments
}

