package com.xebialabs.deployit.core.sql

trait QueryBuilder {
  def query: String

  def parameters: Seq[Any]
}

abstract class AbstractQueryBuilder(implicit override val schemaInfo: SchemaInfo) extends Pageable with QueryBuilder {
  protected[sql] var alias: Option[String] = None

  protected var orderBy: Seq[OrderBy] = Seq()

  def as(alias: String): this.type = {
    this.alias = Some(alias)
    this
  }

  def orderBy(orderBy: OrderBy): this.type = {
    this.orderBy = this.orderBy :+ orderBy
    this
  }

  protected[sql] def orderByFragment(implicit schemaInfo: SchemaInfo) = new OrderByFragment(orderBy.map(_.build(alias)))

  def getOrderBy: Seq[OrderBy] = orderBy

  def clearOrderBy(): Unit = this.orderBy = Seq()
}

object OrderBy {
  def asc(column: Selectable) = AscOrdering(column)
  def desc(column: Selectable) = DescOrdering(column)
}

sealed trait OrderBy {
  val column: Selectable
  def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String
}

case class AscOrdering(override val column: Selectable) extends OrderBy {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} asc"
}

case class DescOrdering(override val column: Selectable) extends OrderBy {
  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = s"${column.build(tableAlias)} desc"
}

object Paging {
  def page(page: Int, size: Int) = Page(page, size)
}

sealed trait Paging

case class Page(page: Int, size: Int) extends Paging

abstract class AbstractUnionBuilder(unionType: String, selects: AbstractQueryBuilder*)(implicit override val schemaInfo: SchemaInfo) extends AbstractQueryBuilder {
  override def parameters: Seq[Any] = selects.flatMap(_.parameters)

  override def query: String = addPaging(selects.map(_.query).mkString(unionType).concat(orderByFragment.build))
}

class UnionBuilder(selects: AbstractQueryBuilder*)(implicit override val schemaInfo: SchemaInfo) extends AbstractUnionBuilder(" union ", selects: _*) {
  override def orderBy(orderBy: OrderBy): this.type = {
    selects.foreach {
      case selectBuilder: SelectBuilder =>
        selectBuilder.select(orderBy.column)
      case joinBuilder: JoinBuilder =>
        joinBuilder.tables.foreach(_.select(orderBy.column))
      case _ =>
        throw new IllegalStateException("Cannot add non-select and non-join query to union")
    }
    super.orderBy(orderBy)
  }
}

class UnionAllBuilder(selects: AbstractQueryBuilder*)(implicit override val schemaInfo: SchemaInfo) extends AbstractUnionBuilder(" union all ", selects: _*)


