package ai.digital.deploy.scheduler

import ai.digital.deploy.scheduler.XldScheduler.Messages.{ScheduleJob, TriggerFired}
import org.apache.pekko.actor.{Actor, ActorLogging, Cancellable, Props}

import java.lang.Math.min
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit
import scala.concurrent.duration._
import scala.language.postfixOps

object XldScheduler {
  def props(jobs: List[ScheduledJob]) : Props = Props(new XldScheduler(jobs))

  object Messages {
    case class ScheduleJob(job: ScheduledJob)
    case class TriggerFired(job: ScheduledJob, lastExecution: LocalDateTime, isMultiPart: Boolean)
  }
}


class XldScheduler(jobs: List[ScheduledJob]) extends Actor with ActorLogging {
  import context._

  var scheduledJobs = scala.collection.mutable.Map[String, Cancellable]()

  // Scala scheduler uses integer internally thus limiting the maximum delay possible
  val MAXIMUM_SCHEDULER_DELAY: Long = Int.MaxValue.toLong * 1000 - 1000

  override def preStart(): Unit = {
    log.info("Singleton Scheduler elected and scheduling jobs!")
    jobs.foreach(job => self ! ScheduleJob(job))
  }

  override def receive: Receive = {
    case ScheduleJob(job) => scheduleJob(job)
    case TriggerFired(job, executionDate, isMultiPart) if executionDate.isAfter(now()) && isMultiPart => scheduleOnce(job, now())
    case TriggerFired(job, _, _) => executeJobAndReschedule(job, now())
  }

  private def scheduleJob(job: ScheduledJob): Unit = {
    log.debug(s"Scheduling job ${job.id} with trigger ${job.trigger}")
    scheduleOnce(job, now())
  }

  private def executeJobAndReschedule(job: ScheduledJob, lastExecution: LocalDateTime): Unit = {
    log.info(s"Trigger ${job.trigger} has fired, executing job ${job.id}")
    job.execute()
    scheduleOnce(job, lastExecution)
  }

  private def scheduleOnce(job: ScheduledJob, lastExecution: LocalDateTime): Unit = {
    val nextExecution = job.trigger.nextExecution(lastExecution)
    log.debug(s"Next job ${job.id} execution scheduled at: $nextExecution")
    val delay = min(ChronoUnit.MILLIS.between(now(),nextExecution), MAXIMUM_SCHEDULER_DELAY)
    val cancellable = context.system.scheduler.scheduleOnce(delay millis, self, TriggerFired(job, nextExecution, delay == MAXIMUM_SCHEDULER_DELAY))
    val key = "job.id-" + nextExecution.toString
    scheduledJobs(key) = cancellable
  }

  private def cancelJob(jobId: String): Unit = {
    scheduledJobs.remove(jobId) match {
      case Some(job) =>
        log.debug(s"Cancelling scheduled job $jobId")
        job.cancel()
        if (job.isCancelled) {
          log.debug(s"Scheduled job $jobId was cancelled successfully")
        } else {
          log.warning(s"Unable to cancel job $jobId. Please restart server for the changes to take effect")
        }
      case None => log.debug(s"No job $jobId currently scheduled. Ignoring...")
    }
  }

  def now(): LocalDateTime =  LocalDateTime.now()


  override def postStop(): Unit = scheduledJobs.keys.foreach(cancelJob)
}
