package ai.digital.deploy.scheduler

import ai.digital.configuration.central.deploy.{ClusterProperties, ServerSideProperties}
import akka.actor.{ActorSystem, PoisonPill}
import akka.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings}
import com.xebialabs.deployit.engine.spi.event.{SpringInstantiatedListener, SystemStartedEvent}
import com.xebialabs.xlplatform.cluster.{ClusterMode, XlCluster}
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.scheduling.annotation.Async
import org.springframework.stereotype.Component

@Component
class SchedulerInitializer extends SpringInstantiatedListener with Logging {

  @Autowired
  var serverSideProperties: ServerSideProperties = _

  @Autowired
  var clusterSideProperties: ClusterProperties = _

  @Autowired
  var scheduledJobsConfiguration: ScheduledJobsConfiguration = _

  def isSchedulerEnabled = serverSideProperties.singletonScheduler.enabled

  def jobs = scheduledJobsConfiguration.getScheduledJobs()

  def clusterMode = clusterSideProperties.mode

  @Async @Subscribe def receiveSystemStart(event: SystemStartedEvent): Unit = {
    if (isSchedulerEnabled)
      startScheduler()
    else
      logger.warn("Singleton Scheduler is disabled.Some deploy jobs may not get triggered")
  }


  private def startScheduler() = {
    clusterMode match {
      /* Scheduler as Cluster Singleton is functional only for Hot-StandBy and Full modes, for Standalone/default we will consider
      it has a non-clustered mode and Scheduler will be functional in every node */
      case ClusterMode.STANDALONE =>
        val system = ActorSystem("Xld_Standalone")
        system.actorOf(XldScheduler.props(jobs))
      case ClusterMode.FULL | ClusterMode.HOT_STANDBY =>
        val system = XlCluster.getActorSystem
        system.actorOf(
          ClusterSingletonManager.props (
            singletonProps = XldScheduler.props(jobs),
            terminationMessage = PoisonPill,
            ClusterSingletonManagerSettings(system)
          ), "xlSingletonScheduler"
        )
      case _ => throw new IllegalArgumentException("This extension is only valid with cluster usage.")
    }
  }


}
