package com.xebialabs.deployit.plugin.powershell;

import com.google.common.base.Joiner;
import com.xebialabs.overthere.OverthereConnection;

import java.util.*;

import static java.nio.charset.StandardCharsets.UTF_16;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

public class PowerShellIniConverter extends BasePowershellVarsConverter {
    private final Map<String, List<String>> iniSections = new HashMap<>();
    private final String iniFilePath;

    PowerShellIniConverter(OverthereConnection connection, Map<String, Object> powershellVars, boolean maskPasswords, boolean exposePasswordPropertiesOnHosts, String iniFilePath) {
        super(connection, powershellVars, maskPasswords, exposePasswordPropertiesOnHosts);
        this.iniFilePath = iniFilePath;
    }

    @Override
    protected void setNullVariable(String variableName) {
        add("$" + variableName + " = $null");
    }

    @Override
    protected void setBooleanVariable(String variableName, boolean variableValue) {
        addSection("Variables", variableName, String.valueOf(variableValue));
        add("$" + variableName + " = As-Boolean($ini.Variables.'" + variableName + "')");
    }

    @Override
    protected void setIntegerVariable(String variableName, int variableValue) {
        addSection("Variables", variableName, String.valueOf(variableValue));
        add("$" + variableName + " = As-Int($ini.Variables.'" + variableName + "')");
    }

    @Override
    protected void setLongVariable(String variableName, long variableValue) {
        addSection("Variables", variableName, String.valueOf(variableValue));
        add("$" + variableName + " = As-Long($ini.Variables.'" + variableName + "')");
    }

    @Override
    protected void setStringVariable(String variableName, String variableValue) {
        addSection("Variables", variableName, variableValue);
        add("$" + variableName + " = $ini.Variables.'" + variableName + "'");
    }

    @Override
    protected void setCollectionOfStringsVariable(String variableName, Collection<?> variableValue) {
        int i = 0;
        List<String> variableNames = new ArrayList<>();
        for (Object o : variableValue) {
            String variable = variableName + "_" + i++;
            addSection("Variables", variable, o.toString());
            variableNames.add("$ini.Variables.'" + variable + "'");
        }
        add("$" + variableName + " = @(" + Joiner.on(", ").join(variableNames) + ")");
    }

    private void addSection(String section, String name, String value) {
        List<String> values = iniSections.putIfAbsent(section, new ArrayList<>());
        values.add(name + "=" + value);
    }

    @Override
    protected void startCreateObject(String objectName) {
        iniSections.put(objectName, new ArrayList<>());
        add("$" + objectName + " = New-Object Object");
        add("$__" + objectName + " = $ini." + objectName);
    }


    @Override
    protected void setBooleanProperty(String objectName, String propertyName, boolean propertyValue) {
        iniSections.get(objectName).add(propertyName + "=" + String.valueOf(propertyValue));
        add("$" + propertyName + "AsBool = As-Boolean($__" + objectName + "." + propertyName + ")");
        setNoteProperty(objectName, propertyName, "$" + propertyName + "AsBool");
    }

    @Override
    protected void setIntegerProperty(String objectName, String propertyName, int propertyValue) {
        iniSections.get(objectName).add(propertyName + "=" + String.valueOf(propertyValue));
        add("$" + propertyName + "AsInt = As-Int($__" + objectName + "." + propertyName + ")");
        setNoteProperty(objectName, propertyName, "$" + propertyName + "AsInt");
    }

    @Override
    protected void setStringProperty(String objectName, String propertyName, String propertyValue) {
        iniSections.get(objectName).add(propertyName + "=" + propertyValue);
        setNoteProperty(objectName, propertyName, "$__" + objectName + "." + propertyName + "");
    }

    @Override
    protected void setPasswordProperty(String objectName, String propertyName, String propertyValue) {
        iniSections.get(objectName).add(propertyName + "=" + encodeBase64String(propertyValue.getBytes(UTF_16)));
        setNoteProperty(objectName, propertyName, "([System.Text.Encoding]::Unicode.GetString([System.Convert]::FromBase64String($__" + objectName + "." + propertyName + ")))");
    }

    @Override
    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> propertyValue) {
        int i = 0;
        List<String> variableNames = new ArrayList<>();
        for (Object o : propertyValue) {
            String variableName = propertyName + "_" + i++;
            addSection(objectName, variableName, o.toString());
            variableNames.add("$__" + objectName + "." + variableName);
        }
        setNoteProperty(objectName, propertyName, "@(" + Joiner.on(", ").join(variableNames) + ")");
    }

    @Override
    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> propertyValue) {
        List<String> contents = new ArrayList<>();
        for (Map.Entry<String, String> entry : propertyValue.entrySet()) {
            String variableName = propertyName + "_" + entry.getKey();
            if (entry.getValue() != null) {
                addSection(objectName, variableName, entry.getValue());
                contents.add(entry.getKey() + " = $__" + objectName + "." + variableName);
            } else {
                contents.add(entry.getKey() + " = $null");
            }
        }
        setNoteProperty(objectName, propertyName, "@{" + Joiner.on("; ").join(contents) + "}");
    }

    @Override
    public List<String> convert() {
        add("$ini = Get-IniContent('" + iniFilePath + "')");
        List<String> lines = super.convert();
        return lines;
    }

    public List<String> getConvertedIniContents() {
        List<String> lines = new ArrayList<>();
        for (Map.Entry<String, List<String>> entry : iniSections.entrySet()) {
            lines.add("[" + entry.getKey() + "]");
            lines.addAll(entry.getValue());
            lines.add("");
        }
        return lines;
    }
}
