package com.xebialabs.deployit.plugin.generic.step;

import com.typesafe.config.Config;

import com.xebialabs.deployit.io.copy.CopyUtil;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableFolderArtifact;
import com.xebialabs.deployit.plugin.overthere.CopyStrategyName;
import com.xebialabs.deployit.plugin.overthere.DefaultCopyStrategyName;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.util.StreamFactoryUtil;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.overthere.ssh.SshConnectionType;
import com.xebialabs.xlplatform.config.ConfigurationHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.List;

public abstract class BaseFolderDeploymentStepSupport extends BaseDeploymentStep {

    private static final Logger logger =  LoggerFactory.getLogger(BaseFolderDeploymentStepSupport.class);
    private static final String COPY_STRATEGY_PROPERTY = "copyStrategy";
    private static final List<String> TAR_ARCHIVE_TYPES = java.util.Arrays.asList("tar", "tar.gz", "tar.bz2");

    protected BaseFolderDeploymentStepSupport() {
    }

    protected BaseFolderDeploymentStepSupport(int order, String description, HostContainer container) {
        super(order, description, container);
    }

    protected BaseFolderDeploymentStepSupport(int order, HostContainer container) {
        super(order, null, container);
    }

    abstract BaseDeployableFolderArtifact getBaseDeployableFolderArtifact();

    protected final boolean isDeployableFolderArtifact() {
       return  getBaseDeployableFolderArtifact() != null;
    }


    protected final boolean isForceArchivedForFolder() {
        boolean isForceArchive;
        BaseDeployableFolderArtifact baseDeployableFolderArtifact = getBaseDeployableFolderArtifact();
        if (baseDeployableFolderArtifact != null ) {
            isForceArchive = baseDeployableFolderArtifact.isForceArchive();
            logger.debug("baseDeployableFolderArtifact is not NULL, forceArchive: {}", isForceArchive);
        } else {
            logger.debug("baseDeployableFolderArtifact is NULL, skipping force archive check.");
            isForceArchive = false;
        }
        return isForceArchive;
    }

    protected final boolean isOneByOneCopyStrategy() {
        Host host = container.getHost();
        boolean hasCopyStrategy = host.hasProperty(COPY_STRATEGY_PROPERTY);
        CopyStrategyName strategy = hasCopyStrategy ? (CopyStrategyName) host.getProperty(COPY_STRATEGY_PROPERTY) : null;
        boolean isOneByOne = hasCopyStrategy && CopyStrategyName.OneByOne.equals(strategy);
        boolean isOneByOneDerived = isOneByOne || (strategy == null && !isAutoDetect() && isDefaultStrategyFallBackToOneByOne());
        logger.debug("[isOneByOneCopyStrategy] hasCopyStrategyProps: {}, strategy: {}, isOneByOneFromHost: {}, isOneByOneDerived: {}", hasCopyStrategy, strategy, isOneByOne, isOneByOneDerived);
        return isOneByOneDerived;
    }

    protected final boolean isFolderCanBeDeletedRecursive() {
        Host host = container.getHost();
        if (host.getOs() != null && !OperatingSystemFamily.WINDOWS.equals(host.getOs())) {
            SshConnectionType connectionType = host.getProperty("connectionType");
            return SshConnectionType.SCP.equals(connectionType)
                || SshConnectionType.SU.equals(connectionType)
                || SshConnectionType.SUDO.equals(connectionType)
                || SshConnectionType.INTERACTIVE_SUDO.equals(connectionType);
        } else {
            logger.debug("[isSshHostAndNotSftp] Host OS is Windows or not set, skipping SSH/SFTP check.");
            return false;
        }
    }

    protected final CopyStrategyName getCopyStrategyName(String srcFileName) {
        Host host = container.getHost();
        boolean hasCopyStrategy = host.hasProperty(COPY_STRATEGY_PROPERTY);
        CopyStrategyName strategy = null;
        if (hasCopyStrategy) {
            strategy = host.getProperty(COPY_STRATEGY_PROPERTY);
        }
        if (strategy == null) {
            if (isAutoDetect()){
                strategy = getAutoDetectCopyStrategyName(srcFileName);
                logger.debug("[getCopyStrategyName] Auto-detected strategy: {}", strategy);
            } else {
                strategy = mapDefaultToCopyStrategyName(getDefaultCopyStrategyNameFromArtifact(), container.getHost());
                logger.debug("[getCopyStrategyName] copy strategy found, using default strategy: {}", strategy);
            }
        }
        logger.debug("[getCopyStrategyName] srcFileName: {}, hasCopyStrategy: {}, strategy: {}", srcFileName, hasCopyStrategy, strategy);
        return strategy;
    }

    private CopyStrategyName getAutoDetectCopyStrategyName(String srcFileName) {
        try {
            String archiveType = StreamFactoryUtil.fetchStreamFactory().getArchiveType(srcFileName);
            
            switch (archiveType) {
                case "tar":
                case "tar.gz":
                case "tar.bz2":
                    return CopyStrategyName.Tar;
                case "zip":
                case "jar":
                    OperatingSystemFamily os = container.getHost().getOs();
                    if (OperatingSystemFamily.WINDOWS.equals(os)) {
                        return CopyStrategyName.ZipWindows;
                    } else if (OperatingSystemFamily.UNIX.equals(os) || OperatingSystemFamily.ZOS.equals(os)) {
                        return CopyStrategyName.ZipUnix;
                    } else {
                        logger.warn("folder:deploy:archiveType not recognized for file {}. Falling back to default strategy.", srcFileName);
                        return mapDefaultToCopyStrategyName(getDefaultCopyStrategyNameFromArtifact(), container.getHost());
                    }
                default:
                    return mapDefaultToCopyStrategyName(getDefaultCopyStrategyNameFromArtifact(), container.getHost());
            }
        } catch (Exception ex) {
            logger.warn("Exception occurred while detecting copy strategy for file: {}. Falling back to default strategy.", srcFileName, ex);
            return mapDefaultToCopyStrategyName(getDefaultCopyStrategyNameFromArtifact(), container.getHost());
        }
    }

    private Boolean isDefaultStrategyFallBackToOneByOne() {
        DefaultCopyStrategyName defaultCopyStrategyName = getDefaultCopyStrategyNameFromArtifact();
        return DefaultCopyStrategyName.OneByOne.equals(defaultCopyStrategyName);
    }

    private CopyStrategyName getDefaultCopyStrategyName(CopyStrategyName detectedCopyStrategyName) {
        if(isAutoDetect()) {
            return detectedCopyStrategyName;
        } else {
            return mapDefaultToCopyStrategyName(getDefaultCopyStrategyNameFromArtifact(), container.getHost());
        }
    }

    private DefaultCopyStrategyName getDefaultCopyStrategyNameFromArtifact() {
        Artifact artifact = this.getBaseDeployableFolderArtifact();
        logger.debug("[getDefaultCopyStrategyNameFromArtifact] Called for artifact: {}", artifact != null ? artifact.getId() : null);
        BaseDeployableFolderArtifact deployable = getBaseDeployableFolderArtifact();
        DefaultCopyStrategyName configuredDefaultCopyStrategyName = DefaultCopyStrategyName.OneByOne;
        if (deployable != null) {
            if (deployable.getDefaultCopyStrategy() != null && deployable.hasProperty("defaultCopyStrategy")) {
                DefaultCopyStrategyName defaultCpyStrategyName = deployable.getDefaultCopyStrategy().getProperty("defaultCopyStrategy");
                if (defaultCpyStrategyName != null) {
                    configuredDefaultCopyStrategyName =  defaultCpyStrategyName;
                    logger.debug("[getDefaultCopyStrategyNameFromArtifact] Found defaultCopyStrategy property: {} on deployable: {}", defaultCpyStrategyName, deployable.getId());
                }  else {
                    logger.debug("[getDefaultCopyStrategyNameFromArtifact] defaultCopyStrategy property is NULL on deployable: {} going for onebyone", deployable.getId());
                }
            } else {
                logger.debug("[getDefaultCopyStrategyNameFromArtifact] No defaultCopyStrategy property found on deployable: {}", deployable.getId());
            }
        } else {
            logger.debug("[getDefaultCopyStrategyNameFromArtifact] Deployable is not BaseDeployableFolderArtifact");
        }
        logger.debug("[getDefaultCopyStrategyNameFromArtifact] Returning result: {}", configuredDefaultCopyStrategyName);
        return configuredDefaultCopyStrategyName;
    }

    private Boolean isAutoDetect() {
        Map<String,String> backupConfig = (Map<String, String>) getCtx().getAttribute("config");
        String autodetectSetting = "deploy.task.artifact-copy-strategy.autodetect";
        Config config = ConfigurationHolder.get();
        boolean allowDetect = false;
        if (config == null) {
            logger.debug("[isAutoDetect] config is null, using backupConfig for autodetect setting");
            allowDetect = Boolean.parseBoolean(backupConfig.getOrDefault(autodetectSetting, "false"));
            logger.debug("[isAutoDetect] autodetect from backupConfig: {}", allowDetect);
        } else {
            if (config.hasPath(autodetectSetting)) {
                allowDetect = config.getBoolean(autodetectSetting);
            }
            logger.debug("[isAutoDetect] autodetect from config: {}", allowDetect);
        }
        logger.debug("[isAutoDetect] Returning: {}", allowDetect);
        return allowDetect;
    }

    private CopyStrategyName mapDefaultToCopyStrategyName(DefaultCopyStrategyName defaultName, Host host) {
        CopyStrategyName result;
        if (defaultName == null) {
            result = CopyStrategyName.OneByOne;
        } else if (defaultName == DefaultCopyStrategyName.OneByOne) {
            result = CopyStrategyName.OneByOne;
        } else if (defaultName == DefaultCopyStrategyName.Zip) {
            if (host != null && host.getOs() != null && OperatingSystemFamily.WINDOWS.equals(host.getOs())) {
                result = CopyStrategyName.ZipWindows;
            } else {
                result = CopyStrategyName.ZipUnix;
            }
        } else if (defaultName == DefaultCopyStrategyName.Tar) {
            result = CopyStrategyName.Tar;
        } else {
            result = CopyStrategyName.OneByOne;
        }
        logger.debug("[mapDefaultToCopyStrategyName] Mapping DefaultCopyStrategyName '{}' with host OS '{}' to CopyStrategyName '{}'", defaultName, host != null ? host.getOs() : null, result);
        return result;
    }


    protected LocalFile folderAsTar(LocalFile localFile) {
        if(isForceArchivedForFolder()) {
            return CopyUtil.zipWithDirectoryAndExcludeRootDir(CopyStrategyName.Tar, localFile);
        } else {
            return CopyUtil.zipWithDirectoryAndRootDir(CopyStrategyName.Tar, localFile);
        }
    }
}
