package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.flow.*;
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.context.PreviewExecutionContext;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.mail.SmtpServer;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.steps.ContextHelper;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import org.slf4j.MDC;

import jakarta.mail.MessagingException;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;

@SuppressWarnings("serial")
@StepMetadata(name = "manual")
public class InstructionStep implements Step, PreviewStep {

    private static final String MDC_KEY_TEMPLATE_PATH = "templatePath";

    @StepParameter(description = "The execution order of the step")
    private Integer order;

    @StepParameter(description = "Description of this step, as it should appear in generated deployment plans")
    private String description;

    @StepParameter(name = "freemarkerContext", description = "Dictionary that contains all values available in the template", required = false, calculated = true)
    private Map<String, Object> vars = new HashMap<>();

    @StepParameter(name = "messageTemplate", description = "The path to the message template to display to the user and/or send out", required = false)
    private String templatePath;

    @StepParameter(name = "mailTo", description = "The list of email receivers to send instructions to", required = false)
    private List<String> toAddresses = newArrayList();

    @StepParameter(description = "The email subject line", required = false, calculated = true)
    private String subject;

    @StepParameter(name = "mailFrom", description = "The email's sender ('From:') email address", required = false, calculated = true)
    private String fromAddress;

    @StepParameter(description = "Mail server that is used to send emails", required = false, calculated = true)
    private SmtpServer mailServer;

    private boolean paused = false;

    private Preview preview;

    private HostContainer container;

    @SuppressWarnings("UnusedDeclaration")
    public InstructionStep() {
    }

    public InstructionStep(int order, String description, HostContainer container, Map<String, Object> vars, String templatePath) {
        this.order = order;
        this.description = description;
        this.vars = newHashMap(vars);
        this.vars.put("step", this);
        this.templatePath = templatePath;
        this.toAddresses = newArrayList();
        this.container = container;
        Objects.requireNonNull(templatePath);
    }

    @RulePostConstruct
    protected void doPostConstruct(final StepPostConstructContext ctx) {
        // calculate default values
        if (!toAddresses.isEmpty() && mailServer == null) {
            mailServer = SmtpServer.getMailServer((Environment) ctx.getDeployedApplication().getEnvironment(), ctx.getRepository());
        }

        if (mailServer != null && isNullOrEmpty(fromAddress)) {
            fromAddress = mailServer.getFromAddress();
        }

        checkMailProperties();

        vars = ContextHelper.defaultContext(ctx, vars);
        if (subject == null) subject = description;
    }

    protected void checkMailProperties() {
        if (!(mailServer == null && isNullOrEmpty(fromAddress) && toAddresses.isEmpty()) &&
                !(mailServer != null && !isNullOrEmpty(fromAddress) && !toAddresses.isEmpty())) {
            throw new IllegalArgumentException(format("To send e-mail the parameters [message-server, mail-from, mail-to] must be set."));
        }
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_TEMPLATE_PATH, templatePath);
        try {
            HashMap<String, Object> variables = newHashMap(vars);
            variables.put("context", ctx);
            String instructions = renderEmailTemplate(templatePath, variables);
            if (!paused) {
                sendInstructions(ctx, instructions);
                return pauseExecution();
            } else {
                return continueExecution(ctx, instructions);
            }
        } finally {
            MDC.remove(MDC_KEY_TEMPLATE_PATH);
        }
    }

    private void sendInstructions(final ExecutionContext ctx, final String instructions) {
        mailInstructions(instructions, ctx);
        ctx.logOutput(instructions);
    }

    private StepExitCode pauseExecution() {
        paused = true;
        return StepExitCode.PAUSE;
    }

    private StepExitCode continueExecution(final ExecutionContext ctx, final String instructions) {
        ctx.logOutput("Assuming manual process performed. Continuing...");
        if (!isNullOrEmpty(instructions)) {
            ctx.logOutput("------------");
            ctx.logOutput("Instructions");
            ctx.logOutput("------------");
            ctx.logOutput(instructions);
        }
        return StepExitCode.SUCCESS;
    }

    protected void mailInstructions(String instructions, ExecutionContext ctx) {
        if (!toAddresses.isEmpty() && mailServer != null) {
            ctx.logOutput("Mailing instructions to " + Joiner.on(',').join(toAddresses));
            subject = (isNullOrEmpty(subject)) ? getDescription() : subject;
            try {
                mailServer.sendMessage(subject, instructions, toAddresses, fromAddress);
            } catch (MessagingException e) {
                ctx.logError("Failed to send mail.", e);
                ctx.logOutput(Strings.repeat("-", 50));
            }
        } else {
            ctx.logOutput("Not sending email instructions.");
        }
    }

    protected String renderEmailTemplate(String template, Map<String, Object> vars) throws IOException, TemplateException {
        if (isNullOrEmpty(template)) {
            return "";
        }
        Configuration cfg = ConfigurationHolder.getConfiguration();
        Template loadedTemplate = cfg.getTemplate(template);
        StringWriter sw = new StringWriter();
        loadedTemplate.setObjectWrapper(new CiAwareObjectWrapper(null, true));
        loadedTemplate.process(vars, sw);
        return sw.toString();
    }

    @Override
    public Preview getPreview() {
        if (preview == null) {
            try {
                HashMap<String, Object> variables = newHashMap(vars);
                variables.put("context", new PreviewExecutionContext());
                preview = Preview.withSourcePathAndContents(templatePath, renderEmailTemplate(templatePath, variables));
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return preview;
    }

    public List<String> getToAddresses() {
        return toAddresses;
    }

    public void setToAddresses(List<String> toAddresses) {
        this.toAddresses = toAddresses;
    }

    public String getSubject() {
        return subject;
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public String getFromAddress() {
        return fromAddress;
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    public SmtpServer getMailServer() {
        return mailServer;
    }

    public void setMailServer(SmtpServer mailServer) {
        this.mailServer = mailServer;
    }
}

