package com.xebialabs.deployit.plugin.generic.deployed;

import java.util.Map;
import java.util.Set;

import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.planning.Noop;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Maps.newHashMap;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.generic.deployed.ArtifactFileUtils.getJavaIoFile;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A script executed on a generic container")
public class ExecutedScript<D extends Deployable> extends AbstractDeployed<D> {

    @Property(required = true, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the create operation.")
    private String createScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the modify operation.")
    private String modifyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the destroy operation.")
    private String destroyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the noop operation.")
    private String noopScript;

    @Property(required = false, hidden = true, description = "Name of working directory on target host. Default is to create a temporary directory which is deleted when connection is closed.")
    private String remoteWorkingDirectoryPath;

    @Property(required = false, hidden = true, defaultValue = "false", description = "Retain the specified working directory on target host after completion.")
    private boolean retainRemoteWorkingDirectory = false;

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the script.")
    private Set<String> classpathResources = newHashSet();

    @Property(hidden = true, required = false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the script."
        +
        "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> templateClasspathResources = newHashSet();

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb, CheckpointInfo checkpoint, Set<String> stepOptions) {
        return addStep(ctx, order, script, verb, checkpoint, stepOptions, null);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb, CheckpointInfo checkpoint, Set<String> stepOptions, Deployed<?, ?> previousDeployed) {
        if (!Strings.nullToEmpty(script).trim().isEmpty()) {
            Map<String, Object> freeMarkerContext = getDeployedAsFreeMarkerContext();
            if(previousDeployed != null) {
                freeMarkerContext = newHashMap(freeMarkerContext);
                freeMarkerContext.put("previousDeployed", previousDeployed);
            }

            ScriptExecutionStep step = new ScriptExecutionStep(order, script, getContainer(), freeMarkerContext, getDescription(verb));
            OverthereFile file = getArtifactIfPresent();

            if (file != null && stepOptions.contains(STEP_OPTION_UPLOAD_ARTIFACT_DATA)) {
                if(this instanceof DerivedArtifact) {
                    //use staging mechanism to upload artifact
                    step.setDerivedArtifact((DerivedArtifact) this);
                } else {
                    //use legacy mechanism to upload artifact
                    step.setArtifact(getJavaIoFile(file));
                }
            }

            if (!Strings.isNullOrEmpty(getRemoteWorkingDirectoryPath())) {
                step.setRemoteWorkingDirPath(getRemoteWorkingDirectoryPath());
                step.setRetainRemoteWorkingDirOnCompletion(isRetainRemoteWorkingDirectory());
            }

            if (stepOptions.contains(STEP_OPTION_UPLOAD_TEMPLATE_CLASSPATH_RESOURCES)) {
                step.setTemplateClasspathResources(newArrayList(getTemplateClasspathResources()));
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES)) {
                step.setClasspathResources(newArrayList(getClasspathResources()));
            }

            if (checkpoint == null) {
                ctx.addStep(step);
            } else {
                ctx.addStepWithCheckpoint(step, checkpoint.delta, checkpoint.operation);
            }

            return true;
        } else {
            return false;
        }
    }

    protected OverthereFile getArtifactIfPresent() {
        if (getDeployable() instanceof DeployableArtifact) {
            return ((DeployableArtifact) getDeployable()).getFile();
        }
        return null;
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, getCreateOrder(), getCreateScript(), getCreateVerb(), checkpoint(delta, Operation.CREATE), getCreateOptions());
    }

    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        boolean modifyStepAdded = addStep(ctx, getModifyOrder(), getModifyScript(), getModifyVerb(), checkpoint(delta, Operation.MODIFY), getModifyOptions(), delta.getPrevious());
        if (!modifyStepAdded) {
            executeDestroy(ctx, delta);
            executeCreate(ctx, delta);
        }
    }

    @SuppressWarnings("unchecked")
    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        ExecutedScript<D> which = (ExecutedScript<D>) delta.getPrevious();
        which.addStep(ctx, getDestroyOrder(), getDestroyScript(), getDestroyVerb(), checkpoint(delta, Operation.DESTROY), getDestroyOptions(), which);
    }

    @Noop
    public void executeNoop(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, getNoopOrder(), getNoopScript(), getNoopVerb(), null, getNoopOptions(), delta.getPrevious());
    }

    public String getCreateScript() {
        return resolveExpression(createScript);
    }

    public void setCreateScript(String createScript) {
        this.createScript = createScript;
    }

    public String getModifyScript() {
        return resolveExpression(modifyScript);
    }

    public void setModifyScript(String modifyScript) {
        this.modifyScript = modifyScript;
    }

    public String getDestroyScript() {
        return resolveExpression(destroyScript);
    }

    public void setDestroyScript(String destroyScript) {
        this.destroyScript = destroyScript;
    }

    public String getNoopScript() {
        return resolveExpression(noopScript);
    }

    public void setNoopScript(String noopScript) {
        this.noopScript = noopScript;
    }

    public String getRemoteWorkingDirectoryPath() {
        return resolveExpression(remoteWorkingDirectoryPath);
    }

    public void setRemoteWorkingDirectoryPath(String remoteWorkingDirectoryPath) {
        this.remoteWorkingDirectoryPath = remoteWorkingDirectoryPath;
    }

    public boolean isRetainRemoteWorkingDirectory() {
        return retainRemoteWorkingDirectory;
    }

    public void setRetainRemoteWorkingDirectory(boolean retainRemoteWorkingDirectory) {
        this.retainRemoteWorkingDirectory = retainRemoteWorkingDirectory;
    }

    public Set<String> getClasspathResources() {
        return resolveExpression(classpathResources);
    }

    public void setClasspathResources(Set<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public Set<String> getTemplateClasspathResources() {
        return resolveExpression(templateClasspathResources);
    }

    public void setTemplateClasspathResources(Set<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    protected static CheckpointInfo checkpoint(Delta delta) {
        return new CheckpointInfo(delta);
    }

    protected static CheckpointInfo checkpoint(Delta delta, Operation operation) {
        return new CheckpointInfo(delta, operation);
    }

    protected static class CheckpointInfo {
        public final Delta delta;
        public final Operation operation;

        CheckpointInfo(Delta delta) {
            this.delta = delta;
            this.operation = delta.getOperation();
        }

        CheckpointInfo(Delta delta, Operation operation) {
            this.delta = delta;
            this.operation = operation;
        }
    }
}
