package com.xebialabs.deployit.plugin.generic.deployed;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.google.common.io.Files;
import com.google.common.io.PatternFilenameFilter;
import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.DeployedSpecific;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.generic.ci.Folder;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.overthere.OverthereFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static java.lang.String.format;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Scripts in the folder are executed against a Container based on a naming convention")
@Placeholders
public class ExecutedFolder<D extends Folder> extends AbstractDeployed<D> implements DerivedArtifact<D> {

    private Map<String, Object> freeMarkerContext = Collections.singletonMap("deployed", (Object) this);

    @Property(required = false, category = "Placeholders", description = "A key/value pair mapping of placeholders in the deployed artifact to their values. Special values are &lt;ignore&gt; or &lt;empty&gt;")
    @DeployedSpecific
    private Map<String, String> placeholders = newHashMap();

    private OverthereFile placeholderProcessedFile;

    @Property(description = "Name of the executor script that will be executed for each script found in the folder.")
    private String executorScript;

    @Property(description = "Regular expression used to identify a script in the folder.  A successful match should returns a single group to which the rollbackScriptPostfix can be appended" +
            " in order to find the associated rollback script or the script's dependent subfolder.  e.g.([0-9]*-.*)\\.sql")
    private String scriptRecognitionRegex;

    @Property(description = "Regular expression used to identify a rollback script in the folder. A successful match should returns a single group, ie the logical script name. e.g. [0-9]*-.*-rollback\\.sql")
    private String rollbackScriptRecognitionRegex;

    @Property(description = "A script's associated rollback script is derived by using the 1st group identified by the scriptRecognitionRegex and then appending this postfix to it." +
            " e.g give name '01-myscript.sql', regex '([0-9]*-.*)\\.sql' and rollback script postfix '-rollback.sql', we can derive the name of the associated rollback script  to be '01-myscript-rollback.sql'")
    private String rollbackScriptPostfix;

    @Property(hidden = true, defaultValue = "true", description = "If set to true, modified scripts are also executed on a MODIFY or a NOOP.", required = false)
    private boolean executeModifiedScripts;

    @Property(hidden = true, defaultValue = "true", description = "If set to true, rollback scripts for modified scripts are also executed on a MODIFY or a NOOP.", required = false)
    private boolean executeRollbackForModifiedScripts;

    @Property(hidden = true, defaultValue = "true", description = "If set to true, the checkpoint is set after the first script in a folder has been executed. Otherwise the checkpoint is set after the last script in a folder has been executed.", required = false)
    private boolean checkpointAfterFirstScript;

    @Property(hidden = true, defaultValue = "common", description = "Common folder that should be uploaded to the working directory.")
    private String commonScriptFolderName;

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the script.")
    private Set<String> classpathResources = newHashSet();

    @Property(hidden = true, required = false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> templateClasspathResources = newHashSet();

    public String resolveExpression(String expression) {
        return ConfigurationHolder.resolveExpression(expression, freeMarkerContext);
    }

    public String getDescription(String script, String verb) {
        return String.format("%s %s on %s", verb, script, getContainer().getName());
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        Step checkpointStep = addSteps(
                ctx,
                this,
                identifyAndOrderCreateScriptsInFolder(getDerivedArtifactAsFile()),
                getScriptRecognitionRegex(),
                getCreateOptions(),
                getCreateOrder(),
                getCreateVerb()
        );

        if (checkpointStep != null) {
            ctx.addCheckpoint(checkpointStep, delta, Operation.CREATE);
        }
    }

    @SuppressWarnings("unchecked")
    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        ExecutedFolder<D> previousDeployed = (ExecutedFolder<D>) delta.getPrevious();
        File previousFolder = previousDeployed.getDerivedArtifactAsFile();
        File currentFolder = getDerivedArtifactAsFile();

        Step firstCheckpointStep = addSteps(
                ctx,
                previousDeployed,
                previousDeployed.compareAndIdentifyOrderedRollbackScriptsToRunForUpdate(previousFolder, currentFolder),
                previousDeployed.getRollbackScriptRecognitionRegex(),
                previousDeployed.getModifyOptions(),
                previousDeployed.getDestroyOrder(),
                previousDeployed.getDestroyVerb()
        );

        Step secondCheckpointStep = addSteps(
                ctx,
                this,
                compareAndIdentifyOrderedScriptsToRunForUpdate(previousFolder, currentFolder),
                getScriptRecognitionRegex(),
                getModifyOptions(),
                getCreateOrder(),
                getCreateVerb()
        );

        Step checkpointStep;
        if(checkpointAfterFirstScript) {
            checkpointStep = firstCheckpointStep != null ? firstCheckpointStep : secondCheckpointStep;
        } else {
            checkpointStep = secondCheckpointStep != null ? secondCheckpointStep : firstCheckpointStep;
        }

        if (checkpointStep != null) {
            ctx.addCheckpoint(checkpointStep, delta, Operation.MODIFY);
        }
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        Step checkpointStep = addSteps(
                ctx,
                this,
                identifyAndOrderRollbackScriptsInFolder(getDerivedArtifactAsFile()),
                getRollbackScriptRecognitionRegex(),
                getDestroyOptions(),
                getDestroyOrder(),
                getDestroyVerb()
        );

        if (checkpointStep != null) {
            ctx.addCheckpoint(checkpointStep, delta, Operation.DESTROY);
        }
    }

    protected static Step addSteps(DeploymentPlanningContext ctx, ExecutedFolder<?> deployed, List<File> scriptsToRun, String scriptNameRegex, Set<String> stepOptions, int order, String verb) {
        File folder = deployed.getDerivedArtifactAsFile();
        Pattern pattern = Pattern.compile(scriptNameRegex);
        File commonResources = new File(folder, deployed.getCommonScriptFolderName());
        Step checkpointStep = null;
        for (File script : scriptsToRun) {
            ScriptExecutionStep step = newScriptExecutionStep(script.getName(), deployed, order, verb);
            if (commonResources.exists()) {
                step.getFileResources().add(commonResources);
            }
            File dependentSubFolderResource = new File(folder, deployed.extractScriptPrefix(pattern, script.getName()));
            if (dependentSubFolderResource.exists()) {
                step.getFileResources().add(dependentSubFolderResource);
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_TEMPLATE_CLASSPATH_RESOURCES)) {
                step.setTemplateClasspathResources(newArrayList(deployed.getTemplateClasspathResources()));
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES)) {
                step.setClasspathResources(newArrayList(deployed.getClasspathResources()));
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_ARTIFACT_DATA)) {
                step.setArtifact(script);
            }
            ctx.addStep(step);

            if(!deployed.isCheckpointAfterFirstScript() || checkpointStep == null) {
                checkpointStep = step;
            }
        }

        // Return the step to be checkpointed
        return checkpointStep;
    }

    /**
     * This method contains a bug! It's still here just for backwards compatibility.
     * It always sets create order and description with create verb to steps, regardless of step type.
     */
    protected Step addSteps(DeploymentPlanningContext ctx, ExecutedFolder<D> deployed, List<File> scriptsToRun, String scriptNameRegex, Set<String> stepOptions) {
        File folder = deployed.getDerivedArtifactAsFile();
        Pattern pattern = Pattern.compile(scriptNameRegex);
        File commonResources = new File(folder, deployed.getCommonScriptFolderName());
        ScriptExecutionStep checkpointStep = null;
        for (File script : scriptsToRun) {
            ScriptExecutionStep step = newScriptExecutionStep(script.getName(), deployed, getCreateOrder(), getCreateVerb());
            if (commonResources.exists()) {
                step.getFileResources().add(commonResources);
            }
            File dependentSubFolderResource = new File(folder, extractScriptPrefix(pattern, script.getName()));
            if (dependentSubFolderResource.exists()) {
                step.getFileResources().add(dependentSubFolderResource);
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_TEMPLATE_CLASSPATH_RESOURCES)) {
                step.setTemplateClasspathResources(newArrayList(getTemplateClasspathResources()));
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_CLASSPATH_RESOURCES)) {
                step.setClasspathResources(newArrayList(getClasspathResources()));
            }
            if (stepOptions.contains(STEP_OPTION_UPLOAD_ARTIFACT_DATA)) {
                step.setArtifact(script);
            }
            ctx.addStep(step);

            if(!checkpointAfterFirstScript || checkpointStep == null) {
                checkpointStep = step;
            }
        }

        // Return the step to be checkpointed
        return checkpointStep;
    }

    /**
     * This method contains a bug! It's still here just for backwards compatibility.
     * It always sets create order and description with create verb to steps, regardless of step type.
     */
    protected static ScriptExecutionStep newScriptExecutionStep(String script, ExecutedFolder<?> deployed) {
        return new ScriptExecutionStep(deployed.getCreateOrder(), deployed.getExecutorScript(),
                deployed.getContainer(), deployed.freeMarkerContext, deployed.getDescription(script, deployed.getCreateVerb()));
    }

    protected static ScriptExecutionStep newScriptExecutionStep(String script, ExecutedFolder<?> deployed, int order, String verb) {
        return new ScriptExecutionStep(
                order,
                deployed.getExecutorScript(),
                deployed.getContainer(),
                deployed.freeMarkerContext,
                deployed.getDescription(script, verb)
        );
    }

    protected String extractScriptPrefix(Pattern pattern, String scriptName) {
        Matcher matcher = pattern.matcher(scriptName);
        boolean matches = matcher.matches();
        checkArgument(matches && matcher.groupCount() == 1, "Script recognition regular expression '%s' run on script name '%s' must return a single group" +
                " to which the rollbackScriptPostfix can be appended to, to determine the associated rollback script or to find its dependent sub-directory", pattern.pattern(), scriptName);
        return matcher.group(1);
    }

    protected File getDerivedArtifactAsFile() {
        checkNotNull(getFile(), "%s has a null file property", this);
        File ioFile = ArtifactFileUtils.getJavaIoFile(getFile());
        checkNotNull(ioFile, "%s could not resolve artifact", this);
        return ioFile;
    }

    protected List<File> identifyAndOrderCreateScriptsInFolder(final File folder) {
        List<File> scriptsToRun = findScriptsToRun(folder, getScriptRecognitionRegex());
        return Ordering.from(new FilenameComparator()).sortedCopy(scriptsToRun);
    }

    protected List<File> findScriptsToRun(final File folder, final String pattern) {
        File[] scriptsToRun = folder.listFiles(new PatternFilenameFilter(pattern));
        scriptsToRun = scriptsToRun == null ? new File[0] : scriptsToRun;
        return newArrayList(scriptsToRun);
    }

    protected List<File> identifyAndOrderRollbackScriptsInFolder(final File folder) {
        List<File> scriptsToRun = findScriptsToRun(folder,getRollbackScriptRecognitionRegex());
        return Ordering.from(new FilenameComparator()).reverse().sortedCopy(scriptsToRun);
    }

    protected List<File> compareAndIdentifyOrderedScriptsToRunForUpdate(final File previousFolder, final File currentFolder) {
        List<File> previousScripts = identifyAndOrderCreateScriptsInFolder(previousFolder);
        List<File> currentScripts = identifyAndOrderCreateScriptsInFolder(currentFolder);
        return Ordering.from(new FilenameComparator()).sortedCopy(difference(previousScripts, currentScripts, executeModifiedScripts));
    }

    protected List<File> compareAndIdentifyOrderedRollbackScriptsToRunForUpdate(final File previousFolder, final File currentFolder) {
        List<File> previousScripts = identifyAndOrderCreateScriptsInFolder(previousFolder);
        List<File> currentScripts = identifyAndOrderCreateScriptsInFolder(currentFolder);
        Collection<File> missingScriptsInCurrentFolder = difference(currentScripts, previousScripts, executeModifiedScripts && executeRollbackForModifiedScripts);
        List<File> rollbackScripts = newArrayList();
        Pattern pattern = Pattern.compile(getScriptRecognitionRegex());
        for (File missingScript : missingScriptsInCurrentFolder) {
            String rollbackScriptName = extractScriptPrefix(pattern, missingScript.getName()) + getRollbackScriptPostfix();
            File rollbackScript = new File(previousFolder, rollbackScriptName);
            if (rollbackScript.exists()) {
                rollbackScripts.add(rollbackScript);
            }
        }
        return Ordering.from(new FilenameComparator()).reverse().sortedCopy(rollbackScripts);
    }

    protected Collection<File> difference(final List<File> previousFiles, final List<File> currentFiles, final boolean compareContents) {
        Set<FileNameAndContentsEqualityWrapper> wrappedPreviousFiles = transform(previousFiles, compareContents);
        Set<FileNameAndContentsEqualityWrapper> wrappedCurrentFiles = transform(currentFiles, compareContents);
        return Collections2.transform(Sets.difference(wrappedCurrentFiles, wrappedPreviousFiles), (FileNameAndContentsEqualityWrapper input) -> input.getFile());
    }

    protected Set<FileNameAndContentsEqualityWrapper> transform(final List<File> files, final boolean compareContents) {
        return newHashSet(Lists.transform(files, (File input) -> new FileNameAndContentsEqualityWrapper(input, compareContents)));
    }


    public String getScriptRecognitionRegex() {
        return scriptRecognitionRegex;
    }

    public void setScriptRecognitionRegex(String scriptRecognitionRegex) {
        this.scriptRecognitionRegex = scriptRecognitionRegex;
    }

    public String getRollbackScriptRecognitionRegex() {
        return rollbackScriptRecognitionRegex;
    }

    public void setRollbackScriptRecognitionRegex(String rollbackScriptRecognitionRegex) {
        this.rollbackScriptRecognitionRegex = rollbackScriptRecognitionRegex;
    }

    public String getRollbackScriptPostfix() {
        return resolveExpression(rollbackScriptPostfix);
    }

    public void setRollbackScriptPostfix(String rollbackScriptPostfix) {
        this.rollbackScriptPostfix = rollbackScriptPostfix;
    }

    public boolean isExecuteModifiedScripts() {
        return executeModifiedScripts;
    }

    public void setExecuteModifiedScripts(boolean executeModifiedScripts) {
        this.executeModifiedScripts = executeModifiedScripts;
    }

    public boolean isExecuteRollbackForModifiedScripts() {
        return executeRollbackForModifiedScripts;
    }

    public void setExecuteRollbackForModifiedScripts(boolean executeRollbackForModifiedScripts) {
        this.executeRollbackForModifiedScripts = executeRollbackForModifiedScripts;
    }

    public boolean isCheckpointAfterFirstScript() {
        return checkpointAfterFirstScript;
    }

    public void setCheckpointAfterFirstScript(boolean checkpointAfterFirstScript) {
        this.checkpointAfterFirstScript = checkpointAfterFirstScript;
    }

    public String getCommonScriptFolderName() {
        return resolveExpression(commonScriptFolderName);
    }

    public void setCommonScriptFolderName(String commonScriptFolderName) {
        this.commonScriptFolderName = commonScriptFolderName;
    }

    public String getExecutorScript() {
        return resolveExpression(executorScript);
    }

    public void setExecutorScript(String executorScript) {
        this.executorScript = executorScript;
    }

    public Set<String> getClasspathResources() {
        return resolveExpression(classpathResources);
    }

    public void setClasspathResources(Set<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public Set<String> getTemplateClasspathResources() {
        return resolveExpression(templateClasspathResources);
    }

    public void setTemplateClasspathResources(Set<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    @Override
    public D getSourceArtifact() {
        return getDeployable();
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }

    @Override
    public OverthereFile getFile() {
        return placeholderProcessedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.placeholderProcessedFile = file;
    }

    private static class FilenameComparator implements Comparator<File> {
        @Override
        public int compare(File o1, File o2) {
            return o1.getName().compareTo(o2.getName());
        }
    }

    private static class FileNameAndContentsEqualityWrapper {
        private final File file;
        private final boolean compareContents;

        public FileNameAndContentsEqualityWrapper(File file, boolean compareContents) {
            this.file = file;
            this.compareContents = compareContents;
        }

        public File getFile() {
            return file;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            FileNameAndContentsEqualityWrapper that = (FileNameAndContentsEqualityWrapper) o;

            if (!file.getName().equals(that.file.getName()))
                return false;

            if (!compareContents) {
                logger.trace("Not comparing contents of two versions of [{}]", file.getName());
                return true;
            }

            if (file.length() != that.file.length()) {
                logger.trace("Found two versions of [{}] to have a different length: [{}] vs. [{}]", file.getName(), file.length(), that.file.length());
                return false;
            }

            try {
                boolean contentsEqual = Files.equal(file, that.file);
                logger.trace("Found two versions of [{}] to be [{}]", file.getName(), contentsEqual ? "equal" : "not equal");
                return contentsEqual;
            } catch (IOException exc) {
                throw new RuntimeException(format("Cannot compare contents of [%s] and [%s]", file, that.file), exc);
            }
        }

        @Override
        public int hashCode() {
            return file != null ? file.getName().hashCode() : 0;
        }
    }

     private static final Logger logger = LoggerFactory.getLogger(ExecutedFolder.class);

}
