package com.xebialabs.deployit.plugin.file.step;

import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider;
import com.xebialabs.deployit.checksum.DefaultChecksumAlgorithmProviderFactory;
import com.xebialabs.deployit.engine.spi.artifact.resolution.ResolvedArtifactFile;
import com.xebialabs.deployit.io.ArtifactFileUtils;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xldeploy.packager.io.StreamerFactory;
import com.xebialabs.xldeploy.packager.placeholders.PlaceholdersUtil;
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner;
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

public class RescanArtifactStep implements Step {

    private final BaseDeployableArtifact artifact;

    public RescanArtifactStep(BaseDeployableArtifact artifact) {
        this.artifact = artifact;
    }

    @Override
    public int getOrder() {
        return 10;
    }

    @Override
    public String getDescription() {
        return "Rescan artifact " + artifact;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) {
        SourceArtifactScanner scanner = new SourceArtifactScanner(StreamerFactory.defaultMappings());
        ChecksumAlgorithmProvider checksumAlgorithmProvider = DefaultChecksumAlgorithmProviderFactory.defaultComparisonAlgorithm();

        try {
            if (!PlaceholdersUtil.SourceArtifactUtil(artifact).shouldScanPlaceholders()) {
                ctx.logError("Should not rescan artifact " + artifact +
                        ". This CI or doesn't have property `scanPlaceholders` or it is not enabled.");
                return StepExitCode.FAIL;
            }

            String oldChecksum = artifact.getChecksum();
            if (!artifact.getFileUri().startsWith("internal:")) {
                rescanExternalArtifact(scanner, checksumAlgorithmProvider);
            } else {
                ResolvedArtifactFile resolved = ArtifactResolverRegistry.resolve(artifact);
                String fileName = getFileName();
                PlaceholdersUtil.SourceArtifactUtil(artifact)
                        .rescan(scanner, fileName, resolved.openStream(), checksumAlgorithmProvider::getMessageDigest);
            }


            artifact.setRescanned(true);
            ctx.getRepository().update(artifact);

            ctx.logOutput("Successfully re-scanned placeholders " + artifact.getPlaceholders() +
                    " for artifact " + artifact + ".");
            return StepExitCode.SUCCESS;
        } catch (Exception exc) {
            ctx.logError("Cannot rescan artifact " + artifact + ". Please check settings.", exc);
            return StepExitCode.FAIL;
        }
    }

    private void rescanExternalArtifact(SourceArtifactScanner scanner,
                                        ChecksumAlgorithmProvider checksumAlgorithmProvider) throws IOException {
        ArtifactFileUtils.checkFileConditions(artifact);
        if (ArtifactFileUtils.hasRealOrResolvedFile(artifact)) {
            PlaceholdersUtil.SourceArtifactUtil(artifact).rescan(scanner, checksumAlgorithmProvider::getMessageDigest);
        } else {
            ResolvedArtifactFile resolved = ArtifactResolverRegistry.resolve(artifact);
            PlaceholdersUtil.SourceArtifactUtil(artifact)
                    .rescan(scanner, resolved.getFileName(), resolved.openStream(), checksumAlgorithmProvider::getMessageDigest);
            artifact.setFile(LocalFile.valueOf(new File(resolved.getFileName())));
        }
    }

    private boolean isChecksumDifferent(String oldChecksum, String newChecksum) {
        return !oldChecksum.equals(newChecksum);
    }

    private String getFileName() throws URISyntaxException {
        URI uri = new URI(artifact.getFileUri());

        switch (uri.getScheme()) {
            case "http":
            case "https":
                return uri.getPath().substring(uri.getPath().lastIndexOf("/") + 1);
            default:
                return uri.getSchemeSpecificPart();
        }
    }
}
