package com.xebialabs.deployit.plugins.releaseauth.command;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.engine.spi.command.CreateCiCommand;
import com.xebialabs.deployit.engine.spi.command.CreateCisCommand;
import com.xebialabs.deployit.engine.spi.command.UpdateCiCommand;
import com.xebialabs.deployit.engine.spi.command.UpdateCisCommand;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import nl.javadude.t2bus.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

@DeployitEventListener
public class RepositoryCommandListener {
    public static final String ADMIN = "admin";
    public static final String VERIFY_CHECKLIST_PERMISSIONS_ON_CREATE = "verifyChecklistPermissionsOnCreate";

    @Subscribe(canVeto = true)
    public void checkWhetherCreateIsAllowed(CreateCiCommand command) {
        if(checkCisNessessary(newArrayList(command.getCi()))) {
            checkCis(command.getCi(),
                    getSatisfactionProperties(getPropertyDescriptors(command.getCi())),
                    newHashSet(command.getRoles()),
                    command.getUsername());
        }
    }

    @Subscribe(canVeto = true)
    @SuppressWarnings("unused")
    public void checkWhetherCreateIsAllowed(CreateCisCommand command) {
        if(checkCisNessessary(command.getCis())) {
            for (ConfigurationItem ci : command.getCis()) {
                checkCis(ci,
                        getSatisfactionProperties(getPropertyDescriptors(ci)),
                        newHashSet(command.getRoles()),
                        command.getUsername());
            }
        }
    }

    @Subscribe(canVeto = true)
    public void checkWhetherUpdateIsAllowed(UpdateCiCommand command) {
        checkCis(command.getUpdate().getNewCi(),
                getChangedProperties(command.getUpdate(), getSatisfactionProperties(getPropertyDescriptors(command.getUpdate().getNewCi()))),
                newHashSet(command.getRoles()),
                command.getUsername());
    }

    @Subscribe(canVeto = true)
    @SuppressWarnings("unused")
    public void checkWhetherUpdateIsAllowed(UpdateCisCommand command) {
        for (Update update : command.getUpdates()) {
            checkCis(update.getNewCi(),
                    getChangedProperties(update, getSatisfactionProperties(getPropertyDescriptors(update.getNewCi()))),
                    newHashSet(command.getRoles()),
                    command.getUsername());
        }
    }

    private static void checkCis(final ConfigurationItem newCi, Supplier<Stream<PropertyDescriptor>> changed,
                                 final Set<String> roles, final String username) {
        if (!newCi.getType().isSubTypeOf(Type.valueOf(Version.class))) {
            return;
        }

        final Descriptor descriptor = newCi.getType().getDescriptor();
        Supplier<Stream<PropertyDescriptor>> propertiesWithRolesLimitations = getPropertiesWithRolesLimitations(newCi, changed);
        Supplier<Stream<PropertyDescriptor>> noPermission = getNoPermission(newCi, roles, username, descriptor, propertiesWithRolesLimitations);

        if (noPermission.get().findAny().isPresent()) {
            logger.error("User [{}] tried to update release conditions {} but didn't have permission for {} on ci {}",
                    username, changed, noPermission, newCi.getId());
            throw new CannotSetReleaseConditionsException(getPropertyNames(noPermission.get()), newCi);
        }

        if (changed.get().findAny().isPresent()) {
            logger.info("User [{}] has updated release conditions {} on ci {}", username, changed, newCi.getId());
        }
    }

    private static List<String> getPropertyNames(Stream<PropertyDescriptor> satisfiesForWhichUserHasNoPermission) {
        return satisfiesForWhichUserHasNoPermission.map(PropertyDescriptor::getName).collect(Collectors.toList());
    }

    @SuppressWarnings("unchecked")
    private static Supplier<Stream<PropertyDescriptor>> getNoPermission(final ConfigurationItem ci,
                                                                        final Set<String> roles,
                                                                        final String username,
                                                                        final Descriptor descriptor,
                                                                        Supplier<Stream<PropertyDescriptor>> properties) {
        return () -> properties.get().filter(propertyDescriptor -> {
            PropertyDescriptor roleProperty = descriptor.getPropertyDescriptor(getRolePropertyName(propertyDescriptor));
            Set<String> requiredRoles = (Set<String>) roleProperty.get(ci);
            return !username.equals(ADMIN) && Sets.intersection(roles, newHashSet(requiredRoles)).isEmpty();
        });
    }

    private static Supplier<Stream<PropertyDescriptor>> getPropertiesWithRolesLimitations(final ConfigurationItem ci,
                                                                                          Supplier<Stream<PropertyDescriptor>> properties) {
        return () -> properties.get().filter(propertyDescriptor -> ci.hasProperty(getRolePropertyName(propertyDescriptor)));
    }

    private static Supplier<Stream<PropertyDescriptor>> getChangedProperties(final Update update,
                                                                             Supplier<Stream<PropertyDescriptor>> properties) {
        return () -> properties.get().filter(input ->
                update.getPreviousCi() == null || !input.areEqual(update.getPreviousCi(), update.getNewCi())
        );
    }

    private static Supplier<Stream<PropertyDescriptor>> getSatisfactionProperties(Collection<PropertyDescriptor> properties) {
        return () -> properties.stream().filter(input -> input.getName().startsWith("satisfies"));
    }

    private static String getRolePropertyName(final PropertyDescriptor property) {
        String propName = property.getName().substring("satisfies".length());
        return "roles" + propName;
    }

    private Collection<PropertyDescriptor> getPropertyDescriptors(ConfigurationItem ci) {
        final Descriptor descriptor = ci.getType().getDescriptor();
        return descriptor.getPropertyDescriptors();
    }

    private boolean checkCisNessessary(List<ConfigurationItem> cis) {
        Version version = (Version) cis
                .stream()
                .filter(input -> input.getType().instanceOf(Type.valueOf(Version.class)))
                .findFirst()
                .orElse(null);

        if (version != null) {
            Application application = version.getApplication();
            if (application != null && application.hasProperty(VERIFY_CHECKLIST_PERMISSIONS_ON_CREATE)) {
                return application.getProperty(VERIFY_CHECKLIST_PERMISSIONS_ON_CREATE);
            }
        }

        return false;
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 403)
    public static class CannotSetReleaseConditionsException extends DeployitException {
        private CannotSetReleaseConditionsException(List<String> properties, ConfigurationItem ci) {
            super("You are not authorized to set release conditions %s on %s", properties, ci);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(RepositoryCommandListener.class);
}
