package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{List => JList}

import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.plugin.context.PreviewExecutionContext
import com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.overthere.{CmdLine, OperatingSystemFamily, OverthereFile}
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}

import scala.jdk.CollectionConverters._
import scala.io.Source.fromInputStream

@StepMetadata(name = "os-script")
class OsScriptStep(context: util.Map[String, Any] = new util.HashMap[String, Any]())
  extends BaseStep with TargetHostSupport with FreemarkerSupport with CopyFileSupport with PreviewStep with StageableStep with SatelliteAware {

  def this() = this(new util.HashMap[String, Any]())

  freemarkerContext = context

  @StepParameter(name = "script", description = "Script base name that will be executed on the target host")
  protected[steps] val scriptBaseName: String = null

  @StepParameter(name = "shell", description = "shell to be used to execute script on the target host", required = false)
  protected[steps] val shell: String = null

  @StepParameter(description = "Resources that are attached to the script and uploaded to the target host", required = false)
  protected[steps] val classpathResources: JList[String] = new util.ArrayList[String]

  @StepParameter(name = "uploadArtifacts", description = "If true, every artifact presented in the freemarker context will be uploaded to the working directory of the script", required = false)
  protected[steps] val uploadArtifactsInFreemarkerContext: Boolean = true

  private var stagedArtifacts: Vector[StagedArtifact] = Vector()

  @StepParameter(name = "preview-script", description = "Path to the Python Preview script to execute (relative to XL Deploy's classpath)", required = false)
  var previewScriptName: String = ""

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    calculateOrder(ctx)
    calculateDescription(ctx)
    calculateTargetHost(ctx)
    calculateFreemarkerContext(ctx)
  }

  override def requestStaging(ctx: StagingContext): Unit = stagedArtifacts = if (uploadArtifactsInFreemarkerContext) {
    freemarkerContext.asScala.toList.collect { case (_, a: Artifact) => StagedArtifact(ctx.stageArtifact(a, targetHost), a) }.toVector
  } else {
    Vector.empty // Nothing to do, please go on.
  }

  override protected[steps] def executeOnTargetHost(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    executeOnTargetHost(scriptBaseName)
  }

  protected[steps] def executeOnTargetHost(_scriptName: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    implicit val scriptName: String = _scriptName
    val artifactFiles = uploadArtifacts
    val targetScriptName = processScript(artifactFiles)
    processClasspathResources(artifactFiles)
    executeScript(targetScriptName)
  }

  private[steps] def processScript(artifactFiles: Map[ArtifactKey, OverthereFile])(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment, scriptName: String) = {
    val resolvedScriptResource = resolveScriptResource(scriptName)
    processAndUploadResource(resolvedScriptResource, artifactFiles)
    resolvedScriptResource.stripSuffix(".ftl")
  }

  private[steps] def processClasspathResources(artifactFiles: Map[ArtifactKey, OverthereFile])
                                              (implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment): Unit =
    Option(classpathResources).getOrElse(new util.ArrayList[String]()).asScala.foreach(processAndUploadResource(_, artifactFiles))

  private[steps] def processAndUploadResource(resourceName: String, artifactFiles: Map[ArtifactKey, OverthereFile])
                                             (implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment): Unit = {
    val targetFile = fileInWorkDir(resourceName.stripSuffix(".ftl"))

    mkTargetDirsIfRequired(targetFile)

    if (resourceName.endsWith(".ftl")) {
      renderFreemarkerTo(resourceName, targetFile, artifactFiles)
    } else {
      copyUrlToTarget(resource(resourceName), targetFile)
    }
  }


  private[steps] def uploadArtifacts(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment): Map[ArtifactKey, OverthereFile] = {
    stagedArtifacts.map { sa =>
      val remoteFile: OverthereFile = OverthereUtils.getUniqueFolder(targetHostEnvironment.workingDir, "xld").getFile(sa.artifact.getFile.getName)
      copyFileToTarget(sa.stagedFile.get(targetHostEnvironment.connection, executionContext), remoteFile)
      (ArtifactKey(sa.artifact), remoteFile)
    }.toMap
  }

  private[steps] def executeScript(scriptResource: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val executable: OverthereFile = fileInWorkDir(scriptResource)
    val configuredShell = if (shell != null) shell else targetHost.getShell
    if (executable.getPath.endsWith(".sh") && configuredShell != null) {
      executeCommand(new CmdLine().addArgument(configuredShell).addArgument(executable.getPath))
    } else {
      executable.setExecutable(true)
      executeCommand(CmdLine.build(executable.getPath))
    }
  }

  private[steps] def resolveScriptResource(): String = {
    resolveScriptResource(scriptBaseName)
  }

  private[steps] def resolveScriptResource(scriptName: String): String = {
    val possibleScriptNames = (targetOs match {
      case OperatingSystemFamily.WINDOWS => List(".bat.ftl", ".cmd.ftl", ".bat", ".cmd", "")
      case _ => List(".sh.ftl", ".sh", "")
    }).map(scriptName + _)

    val resolvedScriptName = possibleScriptNames.find(resources(_).nonEmpty)
    require(resolvedScriptName.isDefined, s"No candidates found for script basename $scriptBaseName")
    resolvedScriptName.getOrElse(throw new NoSuchElementException(s"No candidates found for script basename $scriptBaseName"))
  }

  override lazy val getPreview: Preview = {
    if (previewScriptName.isEmpty) {
      val artifactFiles = stagedArtifacts.map { sa =>
        val previewConnection = PreviewOverthereConnection.getPreviewConnection
        val tempDir = previewConnection.getTempFile("preview", null)
        val remoteFile = previewConnection.getFile(tempDir, sa.artifact.getFile.getName)
        (ArtifactKey(sa.artifact), remoteFile)
      }.toMap
      val scriptPath = resolveScriptResource()
      val scriptContent = if (scriptPath.endsWith(".ftl"))
        processWithFreemarker(scriptPath, maskPassword = true, artifactFiles)
      else
        fromInputStream(resourceAsInputStream(scriptPath)).getLines().mkString("\n")
      Preview.withSourcePathAndContents(scriptPath, scriptContent)
    } else {
      implicit val hostEnvironment: HostEnvironment = HostEnvironment(targetHost)
      try {
        implicit val scriptName: String = previewScriptName
        implicit val context: PreviewExecutionContext = new PreviewExecutionContext
        executeOnTargetHost(scriptName)
        Preview.withSourcePathAndContents(previewScriptName, context.getLog)
      } finally {
        hostEnvironment.connection.close()
      }
    }
  }

  override def getSatellite: Satellite = {
    targetHost.getSatellite
  }

  private case class StagedArtifact(stagedFile: StagedFile, artifact: Artifact) extends Serializable

}
