package com.xebialabs.deployit.plugin.freemarker.sanitizer

import com.xebialabs.deployit.plugin.overthere.HostContainer
import com.xebialabs.overthere.OperatingSystemFamily
import freemarker.template.TemplateMethodModelEx

object OSScriptSanitizer {

  val SANITIZE_EXPRESSION_AUTO: String = "sanitize"
  val SANITIZE_EXPRESSION_UNIX: String = "sanitizeForUnix"
  val SANITIZE_EXPRESSION_WINDOWS: String = "sanitizeForWindows"
  private val WINDOWS_EXTENSIONS = List(".bat.ftl", ".cmd.ftl", ".bat", ".cmd")
  private val UNIX_EXTENSIONS = List(".sh.ftl", ".sh")

  def createOSSpecificSanitizer(templatePath: String, container: HostContainer) = {
    getOSFamily(templatePath, container) match {
      case Some(os) => os match {
        case OperatingSystemFamily.WINDOWS => new WindowsScriptSanitizer
        case _ => new UnixScriptSanitizer
      }
      case None => new NOOPScriptSanitizer
    }
  }

  def getOSFamily(templatePath: String, container: HostContainer): Option[OperatingSystemFamily] = {
    if (templatePath != null && !templatePath.isEmpty)
      return getOSForTemplatePath(templatePath)
    else if (container != null)
      return Option(container.getHost.getOs)
    None
  }

  private def getOSForTemplatePath(templatePath: String): Option[OperatingSystemFamily] = Option {
    WINDOWS_EXTENSIONS.find(templatePath.endsWith(_)) match {
      case Some(x) => OperatingSystemFamily.WINDOWS
      case _ => UNIX_EXTENSIONS.find(templatePath.endsWith(_)) match {
        case Some(x) => OperatingSystemFamily.UNIX
        case _ => null
      }
    }
  }

  def createOSSpecificSanitizer(templatePath: String): TemplateMethodModelEx = createOSSpecificSanitizer(templatePath, null)

  def createUnixSpecificSanitizer(): TemplateMethodModelEx = new UnixScriptSanitizer

  def createWindowsSpecificSanitizer(): TemplateMethodModelEx = new WindowsScriptSanitizer

  def createNOOPSpecificSanitizer(): TemplateMethodModelEx = new NOOPScriptSanitizer

}
