package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{Map => JMap}

import javax.script.ScriptContext
import com.xebialabs.deployit.plugin.api.Deprecations
import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.context.PreviewExecutionContext
import com.xebialabs.deployit.plugin.overthere.Host
import com.xebialabs.deployit.repository.placeholders.PlaceholderRepositoryHolder
import com.xebialabs.deployit.script.SpringBindings
import com.xebialabs.platform.script.jython.ScriptSource.byResource
import com.xebialabs.platform.script.jython._
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}
import com.xebialabs.xlplatform.script.jython.JythonSugarDiscovery
import grizzled.slf4j.Logger
import scala.jdk.CollectionConverters._

import scala.beans.BeanProperty

@StepMetadata(name = "jython")
class JythonStep(@BeanProperty val satellite: Satellite) extends BaseStep with PreviewStep with SatelliteAware {

  def this() = this(null: Satellite)

  def this(host: Host) = this(host.getSatellite)

  @transient private lazy val logger = Logger(getClass)

  @transient private lazy val jython = new JythonSupport {}

  @StepParameter(name = "scriptPath", description = "DEPRECATED: Use 'script' instead. Path to the Python script to execute (relative to XL Deploy's classpath)", required = false)
  private val deprecatedScriptPath: String = ""

  @StepParameter(name = "script", description = "Path to the Python script to execute (relative to XL Deploy's classpath)", required = false)
  @BeanProperty var scriptPath: String = ""

  @StepParameter(description = "Dictionary with keys as variable name and values as variable that are passed to the Python script", required = false, calculated = true)
  @BeanProperty var jythonContext: JMap[String, Any] = new util.HashMap[String, Any]()

  @StepParameter(name = "preview-script", description = "Path to the Python Preview script to execute (relative to XL Deploy's classpath)", required = false)
  @BeanProperty var previewScriptPath: String = ""

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    if (scriptPath.isEmpty) {
      require(deprecatedScriptPath.nonEmpty, "Cannot create step 'jython' since required parameter 'script' is not specified")
      Deprecations.deprecated("**Deprecated** 'script-path' parameter on a jython step is deprecated, use 'script' instead. (Will be removed in XL Deploy 5.5)")
      scriptPath = deprecatedScriptPath
    }

    calculateOrder(ctx)
    calculateDescription(ctx)
    jythonContext = ContextHelper.defaultContext(ctx, jythonContext)
  }

  override def execute(executionContext: ExecutionContext): StepExitCode = {
    try {
      executeScript(executionContext)
    } catch {
      case e: JythonException => handleError(executionContext, e)
    }
  }

  private def executeScript(executionContext: ExecutionContext): StepExitCode = {
    val variables = Bindings.xlDeployApiServices ++ SpringBindings.bindings.get ++ (jythonContext.asScala + ("context" -> executionContext)) +
      ("placeholderRepository" -> PlaceholderRepositoryHolder.getPlaceholderRepository)
    implicit val scriptContext: JythonContext = createJythonContext(executionContext, variables)
    val scriptResult: AnyRef = jython.executeScript(byResource(scriptPath), resultProcessor = extractJythonResult)
    scriptResult match {
      case "PAUSE" => StepExitCode.PAUSE
      case "RETRY" => StepExitCode.RETRY
      case x: Integer if x > 0 =>
        executionContext.logError(s"Python script exited with exit code '$x'")
        StepExitCode.FAIL
      case x: StepExitCode => x
      case _ => StepExitCode.SUCCESS
    }
  }

  private def extractJythonResult(scriptContext: ScriptContext, evaluatedValue: AnyRef): AnyRef = {
    val result = scriptContext.getAttribute("result")
    if(result == null) evaluatedValue else result
  }

  private def createJythonContext(executionContext: ExecutionContext, variables: Map[String, Any]) = {
    JythonContext.withLibrariesAndFactory(
      (Syntactic.loggerLib +: Syntactic.wrapperCodeWithLib(variables.keys)) ++ JythonSugarDiscovery.getExtensionResources
    ) {
      val scriptContext = variables.toScriptContext
      scriptContext.setWriter(new ConsumerWriter(executionContext.logOutputRaw))
      scriptContext.setErrorWriter(new ConsumerWriter(executionContext.logErrorRaw))
      scriptContext
    }
  }

  private def handleError(ctx: ExecutionContext, e: JythonException): StepExitCode = {
    logger.error(e.getMessage, e)
    ctx.logError(e.getMessage, e)
    StepExitCode.FAIL
  }

  override def getPreview: Preview = {
    if (previewScriptPath.nonEmpty) {
      val context = new PreviewExecutionContext()
      val varuables = Bindings.xlDeployApiServices ++ SpringBindings.bindings.get ++ (jythonContext.asScala + ("context" -> context))
      implicit val scriptContext: JythonContext = createJythonContext(context, varuables)
      jython.executeScript(byResource(previewScriptPath), resultProcessor = extractJythonResult)
      Preview.withSourcePathAndContents(previewScriptPath, context.getLog)
    } else {
      Preview.withSourcePathAndContents(scriptPath, byResource(scriptPath).scriptContent)
    }
  }
}
