package ai.digital.deploy.permissions.client.remote

import ai.digital.deploy.permissions.api.rest.dto.{UpdateRoleRequest, _}
import ai.digital.deploy.permissions.api.rest.pagination.{Order, Paging}
import ai.digital.deploy.permissions.api.rest.v1.RolesPaths._
import ai.digital.deploy.permissions.api.rest.v1.{ReferencedPermissionsPaths, RolePrincipalsPaths, RolesPaths}
import ai.digital.deploy.permissions.api.rest.v2.ExtendedRolePrincipalsPaths
import ai.digital.deploy.permissions.client.configuration.PermissionServiceEntityNotFoundError
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, RoleServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.Profile
import org.springframework.core.ParameterizedTypeReference
import org.springframework.http.{HttpEntity, HttpMethod}
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder

import java.util.UUID
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class RemoteRoleServiceClient(@Autowired @Qualifier("permissionServiceRestTemplate") restTemplate: RestTemplate)
    extends RoleServiceClient
    with Logging {

  private val permissionServiceUrl = ServerConfiguration.getInstance().getExternalPermissionServiceUri
  private val roleBaseUrl = permissionServiceUrl + RolesPaths.BASE_PATH
  private val referencedPermissionsBaseUrl = permissionServiceUrl + ReferencedPermissionsPaths.BASE_PATH

  private val getAllRoles = permissionServiceUrl + RolesPaths.BASE_PATH + RolesPaths.READ_ALL_ROLES

  private val createRoleUrl = roleBaseUrl + CREATE_ROLE_PATH
  private val createOrUpdateRoleUrl = roleBaseUrl + CREATE_OR_UPDATE_ROLE_PATH
  private val updateRoleUrl = roleBaseUrl + UPDATE_ROLE
  private val renameRoleUrl = roleBaseUrl + RENAME
  private val deleteRoleUrl = roleBaseUrl + DELETE_ROLE_BY_NAME
  private val deleteRoleByIdUrl = roleBaseUrl + DELETE_ROLE_BY_ID
  private val readRoleUrl = roleBaseUrl + READ_ROLE
  private val readRoleByIdUrl = roleBaseUrl + READ_ROLE_BY_ID
  private val readRolesUrl = roleBaseUrl + READ_ROLES
  private val countRolesUrl = roleBaseUrl + COUNT_ROLES_BY_PATTERN
  private val readRolesForRolePattern = roleBaseUrl + READ_ROLES_BY_PATTERN

  private val createRolePrincipalUrl =
    permissionServiceUrl + RolePrincipalsPaths.BASE_PATH + RolePrincipalsPaths.ADD_PRINCIPALS_PATH

  private val updateRolePrincipalUrl =
    permissionServiceUrl + ExtendedRolePrincipalsPaths.BASE_PATH + ExtendedRolePrincipalsPaths.EDIT_PRINCIPALS_PATH

  private val deleteAllRoleRefsUrl = roleBaseUrl + DELETE_ALL_ROLE_REFS
  private val deleteAllRolesUrl = roleBaseUrl + DELETE_ALL_ROLES

  override def getAll: List[RoleDto] =
    restTemplate.exchange(getAllRoles, HttpMethod.GET, null, ListOfTypeReference[RoleDto]()).getBody

  override def create(name: String, principals: List[String]): RoleWithPrincipalsDto = {
    val role = create(name)
    if (principals.nonEmpty)
      restTemplate
        .exchange(createRolePrincipalUrl,
                  HttpMethod.POST,
                  new HttpEntity(principals.asJava),
                  classOf[RoleWithPrincipalsDto],
                  name
        ).getBody
    else {
      val rolePrincipals = readRolePrincipals(name)
      RoleWithPrincipalsDto(role, rolePrincipals)
    }
  }

  override def createOrUpdate(role: RoleDto): RoleDto = {
    val saveRoleRequest = new UpdateRoleRequest
    saveRoleRequest.setId(role.id.toString)
    saveRoleRequest.setName(role.name)
    restTemplate.exchange(createOrUpdateRoleUrl, HttpMethod.POST, new HttpEntity(saveRoleRequest), classOf[RoleDto]).getBody
  }

  override def update(originalRoleName: String,
                      updatedRoleName: String,
                      principalsToCreate: Set[String],
                      principalsToDelete: Set[String]
  ): RoleWithPrincipalsDto = {
    val request = EditRolePrincipalsRequest(principalsToDelete.toList, principalsToCreate.toList)
    val result = restTemplate
      .exchange(updateRolePrincipalUrl,
                HttpMethod.PATCH,
                new HttpEntity(request),
                classOf[RoleWithPrincipalsDto],
                originalRoleName
      ).getBody
    Option(result.role).orElse(
      throw PermissionServiceRemoteClientError(s"Role with name '${originalRoleName}' doesn't exists - cannot execute update.")
    )
    Option(result.role.id).orElse(
      throw PermissionServiceRemoteClientError(s"Role with name '${originalRoleName}' has no UUID - cannot execute update.")
    )
    if (originalRoleName != updatedRoleName) {
      val updateRoleRequest = new UpdateRoleRequest
      updateRoleRequest.setId(result.role.id.toString)
      updateRoleRequest.setName(updatedRoleName)
      val roleDto =
        restTemplate.exchange(roleBaseUrl, HttpMethod.PUT, new HttpEntity(updateRoleRequest), classOf[RoleDto]).getBody
      RoleWithPrincipalsDto(roleDto, result.principals)
    } else
      result
  }

  override def removeAllReferences(roleName: String): Unit = restTemplate.delete(deleteAllRoleRefsUrl, roleName)

  override def removeAll(): Unit = restTemplate.delete(deleteAllRolesUrl)

  override def create(name: String): RoleDto = {
    val roleRequest = new CreateRoleRequest
    roleRequest.setName(name)
    restTemplate.postForObject(createRoleUrl, roleRequest, classOf[RoleDto])
  }

  override def update(id: UUID, roleName: String): RoleDto = {
    val updateRequest = new UpdateRoleRequest
    updateRequest.id = id.toString
    updateRequest.name = roleName
    restTemplate.postForObject(updateRoleUrl, updateRequest, classOf[RoleDto])
  }

  override def rename(name: String, newName: String): RoleDto = {
    val uri = UriComponentsBuilder.fromUriString(renameRoleUrl).build(name)
    val roleRequest = new CreateRoleRequest
    roleRequest.setName(newName)
    restTemplate.exchange(uri, HttpMethod.PUT, new HttpEntity(roleRequest), classOf[RoleDto]).getBody
  }

  override def delete(roleName: String): Unit = restTemplate.delete(deleteRoleUrl, roleName)

  override def deleteById(roleId: String): Unit = restTemplate.delete(deleteRoleByIdUrl, roleId)

  override def readById(roleId: String): Option[RoleWithPrincipalsDto] =
    Try(restTemplate.getForObject(readRoleByIdUrl, classOf[RoleWithPrincipalsDto], roleId)) match {
      case Success(value) => Option(value)
      case Failure(_: PermissionServiceEntityNotFoundError) => None
      case Failure(exception) => throw exception
    }

  override def read(roleName: String): Option[RoleDto] =
    Try(restTemplate.getForObject(readRoleUrl, classOf[RoleDto], roleName)) match {
      case Success(value) => Option(value)
      case Failure(_: PermissionServiceEntityNotFoundError) => None
      case Failure(exception) => throw exception
    }

  override def read(namePattern: String, page: Int, size: Int, order: SortOrder, field: String): PaginatedResponse[RoleDto] = {
    val uri = UriComponentsBuilder
      .fromUriString(readRolesUrl)
      .queryParam("namePattern", namePattern)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .build().toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, classOf[Array[RoleDto]])
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  private def readRolePrincipals(roleName: String): List[String] = {
    val responseType = new ParameterizedTypeReference[List[String]]() {}
    val readPrincipalsUrl = UriComponentsBuilder
      .fromUriString(
        permissionServiceUrl + RolePrincipalsPaths.BASE_PATH + RolePrincipalsPaths.READ_PRINCIPALS_PATH
      ).build(roleName)
    restTemplate.exchange(readPrincipalsUrl, HttpMethod.GET, null, responseType).getBody
  }

  override def roleExists(roleName: String): Boolean = read(roleName).isDefined

  override def countRoles(roleNamePattern: String): Long = {
    val uri =
      UriComponentsBuilder
        .fromUriString(countRolesUrl)
        .queryParam(RolesPaths.PARAM_ROLE_NAME_PATTERN, roleNamePattern).build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, classOf[Long]).getBody
  }

  override def readByRolePattern(namePattern: String): List[RoleDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleDto]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(readRolesForRolePattern)
        .queryParam(RolesPaths.PARAM_ROLE_NAME_PATTERN, namePattern).build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }
}

case class ListOfTypeReference[T]() extends ParameterizedTypeReference[List[T]]
