package ai.digital.deploy.permissions.client.remote

import ai.digital.deploy.permissions.api.rest.dto.RoleWithReferencedPermissionsDto
import ai.digital.deploy.permissions.api.rest.pagination.{Order, Paging}
import ai.digital.deploy.permissions.api.rest.v1.ReferencedPermissionsPaths
import ai.digital.deploy.permissions.api.rest.v2.ExtendedReferencedPermissionsPaths
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, ReferencedPermissionServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.Profile
import org.springframework.core.ParameterizedTypeReference
import org.springframework.http.{HttpEntity, HttpMethod}
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder

import java.util.UUID
import scala.jdk.CollectionConverters._

@Component
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class RemoteReferencedPermissionServiceClient(
  @Autowired @Qualifier("permissionServiceRestTemplate") restTemplate: RestTemplate
) extends ReferencedPermissionServiceClient {

  private val permissionServiceUrl = ServerConfiguration.getInstance().getExternalPermissionServiceUri
  private val referencePermissionsBaseUrl = permissionServiceUrl + ReferencedPermissionsPaths.BASE_PATH
  private val extendedReferencePermissionBaseUrl = permissionServiceUrl + ExtendedReferencedPermissionsPaths.BASE_PATH
  private val readByRefIdAndRoleNameUrl =
    referencePermissionsBaseUrl + ReferencedPermissionsPaths.READ_REFERENCED_ROLE_PERMISSIONS_PATH
  private val readReferencedPermissionsUrl =
    referencePermissionsBaseUrl + ReferencedPermissionsPaths.READ_REFERENCED_PERMISSIONS
  private val addReferencedPermissionsUrl =
    referencePermissionsBaseUrl + ReferencedPermissionsPaths.ADD_PERMISSIONS_PATH
  private val removeReferencedPermissionsUrl =
    referencePermissionsBaseUrl + ReferencedPermissionsPaths.REMOVE_PERMISSIONS_PATH

  private val getReferencesUrl = extendedReferencePermissionBaseUrl + ExtendedReferencedPermissionsPaths.GET_REFERENCES

  private val readReferencedPermissionsByRolePatternUrl =
    referencePermissionsBaseUrl + ReferencedPermissionsPaths.READ_REFERENCED_PERMISSIONS_BY_ROLE_PATTERN

  override def read(referenceId: UUID, roleName: String): List[String] = {
    val responseType = new ParameterizedTypeReference[List[String]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readByRefIdAndRoleNameUrl)
      .buildAndExpand(referenceId, roleName).toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def read(referenceId: UUID,
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithReferencedPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithReferencedPermissionsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readReferencedPermissionsUrl)
      .queryParam("rolePattern", rolePattern)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .buildAndExpand(referenceId).toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  override def addPermissions(referenceId: UUID, role: String, permissions: List[String]): RoleWithReferencedPermissionsDto = {
    val uri = UriComponentsBuilder
      .fromUriString(addReferencedPermissionsUrl).buildAndExpand(referenceId, role)
      .toUri
    restTemplate.exchange(uri, HttpMethod.POST, new HttpEntity(permissions), classOf[RoleWithReferencedPermissionsDto]).getBody
  }

  override def removePermissions(referenceId: UUID,
                                 roleName: String,
                                 permissions: List[String]
  ): RoleWithReferencedPermissionsDto = {
    val uri = UriComponentsBuilder
      .fromUriString(removeReferencedPermissionsUrl).buildAndExpand(referenceId, roleName)
      .toUri
    restTemplate.exchange(uri, HttpMethod.DELETE, new HttpEntity(permissions), classOf[RoleWithReferencedPermissionsDto]).getBody
  }

  override def getReferencesForRole(roleId: UUID, permissions: List[String]): Set[UUID] = {
    val responseType = new ParameterizedTypeReference[Set[UUID]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getReferencesUrl)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_ROLE_ID, roleId)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_PERMISSIONS, permissions.asJava)
      .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getReferencesForRoles(roleIds: List[UUID], permissions: List[String]): Set[UUID] = {
    val responseType = new ParameterizedTypeReference[Set[UUID]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getReferencesUrl)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_ROLE_IDS, roleIds.asJava)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_PERMISSIONS, permissions.asJava)
      .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getReferencesForRoleName(roleName: String, permissions: List[String]): Set[UUID] = {
    val responseType = new ParameterizedTypeReference[Set[UUID]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getReferencesUrl)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_ROLE_NAME, roleName)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_PERMISSIONS, permissions.asJava)
      .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getReferencesForPrincipal(principalName: String, permissions: List[String]): Set[UUID] = {
    val responseType = new ParameterizedTypeReference[Set[UUID]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getReferencesUrl)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_PRINCIPAL_NAME, principalName)
      .queryParam(ExtendedReferencedPermissionsPaths.PARAM_PERMISSIONS, permissions.asJava)
      .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def readByRolePattern(referenceId: UUID, rolePattern: String): List[RoleWithReferencedPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithReferencedPermissionsDto]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(readReferencedPermissionsByRolePatternUrl)
        .queryParam(ReferencedPermissionsPaths.PARAM_ROLE_PATTERN, rolePattern).build(referenceId)
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }
}
