package ai.digital.deploy.permissions.client.remote

import ai.digital.deploy.permissions.api.rest.dto.RoleWithGlobalPermissionsDto
import ai.digital.deploy.permissions.api.rest.pagination.{Order, Paging}
import ai.digital.deploy.permissions.api.rest.v1.GlobalPermissionsPaths
import ai.digital.deploy.permissions.api.rest.v2.ExtendedGlobalPermissionsPaths
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{GlobalPermissionsServiceClient, PaginatedResponse}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.Profile
import org.springframework.core.ParameterizedTypeReference
import org.springframework.http.{HttpEntity, HttpMethod}
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder

import scala.jdk.CollectionConverters._

@Component
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class RemoteGlobalPermissionsServiceClient(@Autowired @Qualifier("permissionServiceRestTemplate") restTemplate: RestTemplate)
    extends GlobalPermissionsServiceClient {
  // Base paths
  private val permissionServiceUrl = ServerConfiguration.getInstance().getExternalPermissionServiceUri
  private val globalPermissionsBaseUrl = permissionServiceUrl + GlobalPermissionsPaths.BASE_PATH
  private val extendedGlobalPermissionsBaseUrl = permissionServiceUrl + ExtendedGlobalPermissionsPaths.BASE_PATH
  // Endpoints
  private val addPermissionsUrl = globalPermissionsBaseUrl + GlobalPermissionsPaths.ADD_PERMISSIONS_PATH
  private val removePermissionsUrl = globalPermissionsBaseUrl + GlobalPermissionsPaths.DELETE_PERMISSIONS_PATH
  private val readByRoleUrl = globalPermissionsBaseUrl + GlobalPermissionsPaths.READ_ROLE_PERMISSIONS_PATH
  private val read = globalPermissionsBaseUrl + GlobalPermissionsPaths.READ_PERMISSIONS_PATH
  private val readByRolePattern = globalPermissionsBaseUrl + GlobalPermissionsPaths.READ_PERMISSIONS_WITH_FILTER_PATH
  private val readForRoles = globalPermissionsBaseUrl + GlobalPermissionsPaths.READ_PERMISSIONS_FOR_ROLES

  private val getGlobalPermissionsForRoleIdUrl =
    extendedGlobalPermissionsBaseUrl + ExtendedGlobalPermissionsPaths.GET_PERMISSIONS_ROLE_ID
  private val getGlobalPermissionsForPrincipalNameUrl =
    extendedGlobalPermissionsBaseUrl + ExtendedGlobalPermissionsPaths.GET_PERMISSIONS_PRINCIPAL_NAME

  override def addPermissions(roleName: String, permissions: List[String]): RoleWithGlobalPermissionsDto = {
    val uri = UriComponentsBuilder.fromUriString(addPermissionsUrl).build(roleName)
    restTemplate.postForObject(uri, permissions, classOf[RoleWithGlobalPermissionsDto])
  }

  override def removePermissions(roleName: String, permissions: List[String]): RoleWithGlobalPermissionsDto = {
    val uri = UriComponentsBuilder.fromUriString(removePermissionsUrl).build(roleName)
    restTemplate
      .exchange(uri, HttpMethod.DELETE, new HttpEntity(permissions.asJava), classOf[RoleWithGlobalPermissionsDto]).getBody
  }

  override def read(roleName: String): List[String] = {
    val responseType = new ParameterizedTypeReference[List[String]]() {}
    val uri = UriComponentsBuilder.fromUriString(readByRoleUrl).build(roleName)
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def read(rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithGlobalPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithGlobalPermissionsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(read)
      .queryParam("rolePattern", rolePattern)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .build().toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  override def getGlobalPermissionsForRoleId(roleId: String): List[String] = {
    val responseType = new ParameterizedTypeReference[List[String]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getGlobalPermissionsForRoleIdUrl)
      .buildAndExpand(roleId).toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getGlobalPermissionsForPrincipal(principalName: String): Set[String] = {
    val responseType = new ParameterizedTypeReference[Set[String]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getGlobalPermissionsForPrincipalNameUrl)
      .buildAndExpand(principalName).toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getGlobalPermissionsForRoles(roleNames: List[String]): List[RoleWithGlobalPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithGlobalPermissionsDto]]() {}
    restTemplate
      .exchange(readForRoles, HttpMethod.GET, new HttpEntity(roleNames), responseType).getBody
  }

  override def readByRolePattern(rolePattern: String): List[RoleWithGlobalPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithGlobalPermissionsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readByRolePattern)
      .queryParam(GlobalPermissionsPaths.PARAM_ROLE_PATTERN, rolePattern)
      .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }
}
