package ai.digital.deploy.permissions.client.local

import ai.digital.deploy.permissions.api.rest.dto.converters.{RoleConverter, RoleWithGlobalPermissionsConverter}
import ai.digital.deploy.permissions.api.rest.dto.{ReferencedPermissionsDto, RoleWithPermissionsDto}
import ai.digital.deploy.permissions.client.{PaginatedResponse, PermissionServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.service.{GlobalPermissionService, ReferencedPermissionService, RoleService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Service
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalPermissionsServiceClient(
  @Autowired globalPermissionService: GlobalPermissionService,
  @Autowired referencedPermissionService: ReferencedPermissionService,
  @Autowired roleService: RoleService
) extends PermissionServiceClient {
  override def createOrUpdate(referenceIdMaybe: Option[UUID],
                              roleName: String,
                              permissionsToAdd: List[String],
                              permissionToDelete: List[String]
  ): Unit =
    referenceIdMaybe match {
      case Some(referenceId) =>
        referencedPermissionService.edit(referenceId, roleName, permissionsToAdd, permissionToDelete) match {
          case Left(exception) => throw RoleNameNotFoundException(exception.name)
          case Right(_) =>
        }
      case _ => globalPermissionService.edit(roleName, permissionsToAdd, permissionToDelete)
    }

  override def removeForReference(referenceId: UUID): Unit =
    referencedPermissionService.removeForReference(referenceId)

  override def removeAll(): Unit = {
    referencedPermissionService.removeAll()
    globalPermissionService.removeAll()
  }

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(referenceMaybe: Option[UUID], permissions: List[String], allRoles: List[String]): Boolean =
    referenceMaybe
      .map { referenceId =>
        referencedPermissionService.checkPermission(referenceId, permissions, allRoles)
      }
      .getOrElse(globalPermissionService.checkPermission(permissions, allRoles))

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(referenceMaybe: Option[UUID],
                               permissions: List[String],
                               allRoles: List[String],
                               principals: List[String]
  ): Boolean =
    referenceMaybe
      .map { referenceId =>
        referencedPermissionService.checkPermission(referenceId, permissions, allRoles, principals)
      }
      .getOrElse(globalPermissionService.checkPermission(permissions, allRoles, principals))

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(references: List[UUID], permissions: List[String], allRoles: List[String]): Map[String, Boolean] =
    references
      .map(reference => reference.toString -> referencedPermissionService.checkPermission(reference, permissions, allRoles)).toMap

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(references: List[UUID],
                               permissions: List[String],
                               allRoles: List[String],
                               principals: List[String]
  ): Map[String, Boolean] =
    references
      .map(reference =>
        reference.toString -> referencedPermissionService.checkPermission(reference, permissions, allRoles, principals)
      ).toMap

  @ReadOnlyTransactionalPermissionService
  override def getAllPermissionsForRole(roleName: String, page: Int, size: Int): RoleWithPermissionsDto =
    getAllPermissionsForRole(roleName)

  @ReadOnlyTransactionalPermissionService
  override def getAllPermissionsForRoles(names: List[String], page: Int, size: Int): PaginatedResponse[RoleWithPermissionsDto] = {
    val pageable = PageRequest.of(page - 1, size)
    val roles = roleService.read(names, pageable)
    val roleNames = roles.asScala.toList.map(_.name)
    val globalPermissions = globalPermissionService.read(roleNames).groupBy(_.role.name)
    val referencedPermissions = referencedPermissionService.read(roleNames) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(result) => result.groupBy(_.role.name)
    }
    val dtos = roles.asScala.toList.map { role =>
      val referencedPermissionsForRole = referencedPermissions.getOrElse(role.name, List.empty).groupBy(_.reference)
      val referencedPermissionsDto = referencedPermissionsForRole.map { rp =>
        ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
      }.toList
      RoleWithPermissionsDto(
        RoleConverter.roleToDto(role),
        RoleWithGlobalPermissionsConverter.globalPermissionToDto(globalPermissions.getOrElse(role.name, List.empty)),
        referencedPermissionsDto
      )
    }
    PaginatedResponse[RoleWithPermissionsDto](
      dtos,
      roles.getTotalElements,
      page,
      size,
      roles.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def getAllPermissionsForRole(roleName: String): RoleWithPermissionsDto = {
    val globalPermissions = globalPermissionService.read(roleName).map(_.permissionName)
    val referencedPermissions = referencedPermissionService.read(roleName) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(result) => result.groupBy(_.reference)
    }
    val role = roleService.read(roleName).getOrElse(throw RoleNameNotFoundException(roleName))
    val referencedPermissionsDto = referencedPermissions.map { rp =>
      ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
    }.toList
    RoleWithPermissionsDto(RoleConverter.roleToDto(role), globalPermissions, referencedPermissionsDto)
  }

  @ReadOnlyTransactionalPermissionService
  override def getAllPermissionsForRoles(roleNames: List[String]): List[RoleWithPermissionsDto] = {
    val globalPermissions = globalPermissionService.read(roleNames).groupBy(_.role.name)
    val referencedPermissions = referencedPermissionService.read(roleNames) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(permissions) => permissions.groupBy(_.role.name)
    }
    roleService.read(roleNames) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(roles) =>
        roles.map { role =>
          val referencedPermissionsForRole = referencedPermissions.getOrElse(role.name, List.empty).groupBy(_.reference)
          val referencedPermissionsDto = referencedPermissionsForRole.map { rp =>
            ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
          }.toList
          RoleWithPermissionsDto(
            RoleConverter.roleToDto(role),
            RoleWithGlobalPermissionsConverter.globalPermissionToDto(globalPermissions.getOrElse(role.name, List.empty)),
            referencedPermissionsDto
          )
        }
    }
  }
}
