package ai.digital.deploy.permissions.client.configuration

import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import org.apache.http.client.config.RequestConfig
import org.apache.http.conn.ssl.NoopHostnameVerifier
import org.apache.http.impl.client.{HttpClients, SystemDefaultCredentialsProvider}
import org.apache.http.impl.conn.SystemDefaultRoutePlanner
import org.apache.http.ssl.SSLContextBuilder
import org.springframework.boot.web.client.RestTemplateBuilder
import org.springframework.context.annotation.{Bean, Configuration, Profile}
import org.springframework.http.HttpStatus
import org.springframework.http.HttpStatus.Series
import org.springframework.http.client.{ClientHttpResponse, HttpComponentsClientHttpRequestFactory}
import org.springframework.web.client.{ResponseErrorHandler, RestTemplate}

import java.io.File
import java.net.ProxySelector
import java.security.cert.X509Certificate
import java.time.Duration

@Configuration
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class PermissionServiceClientConfiguration(serverConfiguration: ServerConfiguration) {

  @Bean
  def restTemplateResponseErrorHandler(): ResponseErrorHandler =
    new ResponseErrorHandler() {
      override def hasError(httpResponse: ClientHttpResponse): Boolean =
        httpResponse.getStatusCode.series() == Series.CLIENT_ERROR ||
          httpResponse.getStatusCode.series() == Series.SERVER_ERROR

      override def handleError(response: ClientHttpResponse): Unit =
        response.getStatusCode.series() match {
          case Series.SERVER_ERROR =>
            throw PermissionServiceServerError(response.getStatusCode.getReasonPhrase)
          case Series.CLIENT_ERROR if response.getStatusCode == HttpStatus.NOT_FOUND =>
            throw PermissionServiceEntityNotFoundError(response.getStatusCode.getReasonPhrase)
          case Series.CLIENT_ERROR =>
            throw PermissionServiceClientError(response.getStatusCode.getReasonPhrase)
        }

    }

  @Bean
  def permissionServiceRestTemplate(builder: RestTemplateBuilder): RestTemplate = {
    val sslContextBuilder = new SSLContextBuilder
    val httpClientBuilder = HttpClients.custom
    val timeout = Duration.ofMinutes(1).toMillis.toInt

    if (serverConfiguration.isSsl)
      Option(serverConfiguration.getKeyStorePath)
        .map(path =>
          sslContextBuilder.loadTrustMaterial(new File(path),
                                              ServerConfigurationHelper.getKeyStorePassword.map(_.toCharArray).orNull
          )
        )
        .getOrElse(sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true))
    else
      sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true)
    httpClientBuilder.setSSLHostnameVerifier(new NoopHostnameVerifier)
    httpClientBuilder.setRoutePlanner(new SystemDefaultRoutePlanner(ProxySelector.getDefault))
    httpClientBuilder.setDefaultCredentialsProvider(new SystemDefaultCredentialsProvider)
    httpClientBuilder
      .setSSLContext(sslContextBuilder.build)
      .setDefaultRequestConfig(RequestConfig.custom.setSocketTimeout(timeout).setConnectTimeout(timeout).build)
    builder
      .requestFactory(() => new HttpComponentsClientHttpRequestFactory(httpClientBuilder.build()))
      .errorHandler(restTemplateResponseErrorHandler())
      .build()
  }

}

final case class PermissionServiceServerError(message: String) extends DeployitException(message)
final case class PermissionServiceClientError(message: String) extends DeployitException(message)
final case class PermissionServiceEntityNotFoundError(message: String) extends DeployitException(message)
