package ai.digital.deploy.permissions.client.local

import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithPrincipalsDto}
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, RoleServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.exception.{RoleNameAlreadyExistsException, RoleNameNotFoundException}
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.service.{RoleNameAlreadyExistsServiceException, RoleNameNotFoundServiceException, RolePrincipalService, RoleService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Component

import java.util.UUID
import scala.jdk.CollectionConverters._

@Component
@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalRoleServiceClient(
  @Autowired roleService: RoleService,
  @Autowired rolePrincipalService: RolePrincipalService
) extends RoleServiceClient {
  @ReadOnlyTransactionalPermissionService
  override def getAll: List[RoleDto] = roleService.readAll()

  override def create(name: String, principals: List[String]): RoleWithPrincipalsDto = {
    roleService.create(name)
    rolePrincipalService.add(name, principals) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right((role, rolePrincipals)) => RoleWithPrincipalsDto(role, rolePrincipals.map(_.principalName))
    }
  }

  override def createOrUpdate(role: RoleDto): RoleDto =
    roleService.createOrUpdate(Role(role.id, role.name)) match {
      case Left(exception) => throw RoleNameAlreadyExistsException(exception.name)
      case Right(result) => result
    }

  override def update(originalRoleName: String,
                      updatedRoleName: String,
                      principalsToCreate: Set[String],
                      principalsToDelete: Set[String]
  ): RoleWithPrincipalsDto =
    rolePrincipalService.edit(originalRoleName, principalsToCreate.toList, principalsToDelete.toList) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(principalEditResult) =>
        if (originalRoleName == updatedRoleName)
          RoleWithPrincipalsDto(principalEditResult._1, principalEditResult._2.map(_.principalName))
        else
          roleService.rename(originalRoleName, updatedRoleName) match {
            case Left(exception: RoleNameAlreadyExistsServiceException) => throw RoleNameNotFoundException(exception.name)
            case Left(exception: RoleNameNotFoundServiceException) => throw RoleNameAlreadyExistsException(exception.name)
            case Right(updatedRole) =>
              rolePrincipalService.read(updatedRole.name) match {
                case Left(exception) => throw RoleNameNotFoundException(exception.name)
                case Right(principals) => RoleWithPrincipalsDto(updatedRole, principals.map(_.principalName))
              }
          }
    }

  override def removeAllReferences(roleName: String): Unit =
    roleService.deleteAllRoleReferences(roleName) match {
      case Left(exception) => throw RoleNameNotFoundException(exception.name)
      case Right(_) =>
    }

  override def create(name: String): RoleDto =
    roleService.create(name) match {
      case Left(exception) => throw RoleNameAlreadyExistsException(exception.name)
      case Right(result) => result
    }

  override def update(id: UUID, roleName: String): RoleDto =
    roleService.update(Role(id, roleName))

  override def rename(name: String, newName: String): RoleDto =
    roleService.rename(name, newName) match {
      case Left(exception: RoleNameNotFoundServiceException) => throw RoleNameNotFoundException(exception.name)
      case Left(exception: RoleNameAlreadyExistsServiceException) => throw RoleNameAlreadyExistsException(exception.name)
      case Right(result) => result
    }

  override def delete(roleName: String): Unit = roleService.delete(roleName)

  override def deleteById(roleId: String): Unit = roleService.delete(UUID.fromString(roleId))

  @ReadOnlyTransactionalPermissionService
  override def readById(roleId: String): Option[RoleWithPrincipalsDto] =
    roleService.read(UUID.fromString(roleId))

  @ReadOnlyTransactionalPermissionService
  override def read(roleName: String): Option[RoleDto] = roleService.read(roleName)

  @ReadOnlyTransactionalPermissionService
  override def read(namePattern: String, page: Int, size: Int, order: SortOrder, field: String): PaginatedResponse[RoleDto] = {
    val rolePage = roleService.read(namePattern, PageRequest.of(page - 1, size, SortOrder.getSort(order, field)))
    PaginatedResponse[RoleDto](
      rolePage.getContent.asScala.toList,
      rolePage.getTotalElements,
      page,
      size,
      rolePage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(namePattern: String): List[RoleDto] =
    roleService.readByRolePattern(namePattern)

  override def removeAll(): Unit =
    roleService.removeAll()

  @ReadOnlyTransactionalPermissionService
  override def roleExists(roleName: String): Boolean = roleService.read(roleName).isDefined

  @ReadOnlyTransactionalPermissionService
  override def countRoles(roleNamePattern: String): Long =
    roleService.count(roleNamePattern)
}
