package ai.digital.deploy.permissions.client.remote

import ai.digital.deploy.permissions.api.rest.dto.{CheckPermissionRequest, CheckPermissionWithPrincipalsRequest, CheckPermissionWithReferencesAndPrincipalsRequest, CheckPermissionWithReferencesRequest, EditRolePermissionsRequest, RoleWithGlobalPermissionsDto, RoleWithPermissionsDto, RoleWithReferencedPermissionsDto}
import ai.digital.deploy.permissions.api.rest.pagination.Paging
import ai.digital.deploy.permissions.api.rest.v1.PermissionsPaths
import ai.digital.deploy.permissions.api.rest.v2.{ExtendedGlobalPermissionsPaths, ExtendedReferencedPermissionsPaths}
import ai.digital.deploy.permissions.client.{PaginatedResponse, PermissionServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.Profile
import org.springframework.core.ParameterizedTypeReference
import org.springframework.http.{HttpEntity, HttpMethod}
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder

import java.net.URI
import java.util.UUID
import scala.jdk.CollectionConverters._

@Component
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class RemotePermissionServiceClient(@Autowired @Qualifier("permissionServiceRestTemplate") restTemplate: RestTemplate)
    extends PermissionServiceClient {
  private val permissionServiceUrl = ServerConfiguration.getInstance().getExternalPermissionServiceUri
  private val permissionsBaseUrl = permissionServiceUrl + PermissionsPaths.BASE_PATH
  private val referencePermissionsBaseUrl = permissionServiceUrl + ExtendedReferencedPermissionsPaths.BASE_PATH
  private val globalPermissionsBaseUrl = permissionServiceUrl + ExtendedGlobalPermissionsPaths.BASE_PATH
  private val createReferencePermissionUrl =
    referencePermissionsBaseUrl + ExtendedReferencedPermissionsPaths.EDIT_PERMISSIONS_PATH
  private val createGlobalPermissionUrl =
    globalPermissionsBaseUrl + ExtendedGlobalPermissionsPaths.EDIT_ROLE_PERMISSIONS_PATH
  private val deleteReferencePermissionUrl =
    referencePermissionsBaseUrl + ExtendedReferencedPermissionsPaths.REFERENCE_PERMISSIONS_PATH

  private val deleteReferencePermissionsUrl =
    referencePermissionsBaseUrl + ExtendedReferencedPermissionsPaths.DELETE_ALL_PATH
  private val deleteGlobalPermissionsUrl =
    globalPermissionsBaseUrl + ExtendedGlobalPermissionsPaths.DELETE_ALL_PATH

  private val getAllPermissionsForRolePath = permissionsBaseUrl + PermissionsPaths.GET_ALL_PERMISSION_FOR_ROLE
  private val getAllPermissionsForRolesPath = permissionsBaseUrl + PermissionsPaths.GET_ALL_PERMISSION_FOR_ROLES
  private val getAllPermissionsForRolesWithFilterPath =
    permissionsBaseUrl + PermissionsPaths.GET_ALL_PERMISSION_FOR_ROLES_WITH_FILTER

  private val checkPermissionForRolesAndReferencePath =
    permissionsBaseUrl + PermissionsPaths.CHECK_PERMISSION_FOR_ROLES_AND_REFERENCE
  private val checkPermissionForRolesReferenceAndPrincipalsPath =
    permissionsBaseUrl + PermissionsPaths.CHECK_PERMISSION_FOR_ROLES_REFERENCE_AND_PRINCIPALS
  private val checkPermissionForRolesAndReferencesPath =
    permissionsBaseUrl + PermissionsPaths.CHECK_PERMISSION_FOR_ROLES_AND_REFERENCES
  private val checkPermissionForRolesReferencesAndPrincipalsPath =
    permissionsBaseUrl + PermissionsPaths.CHECK_PERMISSION_FOR_ROLES_REFERENCES_AND_PRINCIPALS

  override def createOrUpdate(referenceIdMaybe: Option[UUID],
                              roleName: String,
                              permissionsToAdd: List[String],
                              permissionToDelete: List[String]
  ): Unit =
    referenceIdMaybe match {
      case Some(referenceId) =>
        createOrUpdate(
          UriComponentsBuilder
            .fromUriString(createReferencePermissionUrl).buildAndExpand(referenceId, roleName).toUri,
          EditRolePermissionsRequest(permissionToDelete.asJava, permissionsToAdd.asJava),
          classOf[RoleWithReferencedPermissionsDto]
        )
      case _ =>
        createOrUpdate(
          UriComponentsBuilder
            .fromUriString(createGlobalPermissionUrl).buildAndExpand(roleName).toUri,
          EditRolePermissionsRequest(permissionToDelete.asJava, permissionsToAdd.asJava),
          classOf[RoleWithGlobalPermissionsDto]
        )
    }

  private def createOrUpdate[T](uri: URI, request: EditRolePermissionsRequest, clazz: Class[T]): Unit =
    restTemplate.exchange(uri, HttpMethod.PATCH, new HttpEntity(request), clazz)

  override def removeForReference(referenceId: UUID): Unit =
    restTemplate.delete(deleteReferencePermissionUrl, referenceId)

  override def removeAll(): Unit = {
    restTemplate.delete(deleteReferencePermissionsUrl)
    restTemplate.delete(deleteGlobalPermissionsUrl)
  }

  override def checkPermission(referenceMaybe: Option[UUID], permissions: List[String], allRoles: List[String]): Boolean = {
    val uri =
      UriComponentsBuilder
        .fromUriString(checkPermissionForRolesAndReferencePath)
        .queryParam("referenceMaybe",
                    referenceMaybe match {
                      case Some(reference) => reference.toString
                      case None => null
                    }
        )
        .queryParam("permissions", permissions.asJava)
        .queryParam("allRoles", allRoles.asJava)
        .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, classOf[Boolean]).getBody
  }

  override def checkPermission(referenceMaybe: Option[UUID],
                               permissions: List[String],
                               allRoles: List[String],
                               principals: List[String]
  ): Boolean = {
    val uri =
      UriComponentsBuilder
        .fromUriString(checkPermissionForRolesReferenceAndPrincipalsPath)
        .queryParam("referenceMaybe",
                    referenceMaybe match {
                      case Some(reference) => reference.toString
                      case None => null
                    }
        ).queryParam("permissions", permissions.asJava)
        .queryParam("allRoles", allRoles.asJava)
        .queryParam("principals", principals.asJava)
        .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, classOf[Boolean]).getBody
  }

  override def checkPermission(references: List[UUID],
                               permissions: List[String],
                               allRoles: List[String]
  ): Map[String, Boolean] = {
    val responseType = new ParameterizedTypeReference[Map[String, Boolean]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(checkPermissionForRolesAndReferencesPath)
        .queryParam("references", references.asJava)
        .queryParam("permissions", permissions.asJava)
        .queryParam("allRoles", allRoles.asJava)
        .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def checkPermission(references: List[UUID],
                               permissions: List[String],
                               allRoles: List[String],
                               principals: List[String]
  ): Map[String, Boolean] = {
    val responseType = new ParameterizedTypeReference[Map[String, Boolean]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(checkPermissionForRolesReferencesAndPrincipalsPath)
        .queryParam("references", references.asJava)
        .queryParam("permissions", permissions.asJava)
        .queryParam("allRoles", allRoles.asJava)
        .queryParam("principals", principals.asJava)
        .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def getAllPermissionsForRole(roleName: String, page: Int, size: Int): RoleWithPermissionsDto = ???

  override def getAllPermissionsForRoles(roleNames: List[String],
                                         page: Int,
                                         size: Int
  ): PaginatedResponse[RoleWithPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithPermissionsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getAllPermissionsForRolesWithFilterPath)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam("roleNames", roleNames.asJava)
      .build().toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  override def getAllPermissionsForRole(roleName: String): RoleWithPermissionsDto = {
    val uri = UriComponentsBuilder.fromUriString(getAllPermissionsForRolePath).build(roleName)
    restTemplate.getForObject(uri, classOf[RoleWithPermissionsDto])
  }

  override def getAllPermissionsForRoles(roleNames: List[String]): List[RoleWithPermissionsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithPermissionsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(getAllPermissionsForRolesPath)
      .queryParam("roleNames", roleNames.asJava)
      .build().toUri
    restTemplate
      .exchange(uri, HttpMethod.GET, null, responseType).getBody
  }
}
