package ai.digital.deploy.permissions.client.local

import ai.digital.deploy.permissions.api.rest.dto.RoleWithPrincipalsDto
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleWithPrincipalsConverter._
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, RolePrincipalsServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.service.RolePrincipalService
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Service
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalRolePrincipalsServiceClient(rolePrincipalService: RolePrincipalService) extends RolePrincipalsServiceClient {
  override def addPrincipals(roleName: String, principals: List[String]): RoleWithPrincipalsDto = {
    val (role, rp) = rolePrincipalService.add(roleName, principals)
    RoleWithPrincipalsDto(role, rp.map(_.principalName))
  }

  override def removePrincipals(roleName: String, principals: List[String]): RoleWithPrincipalsDto = {
    val (role, rp) = rolePrincipalService.remove(roleName, principals)
    RoleWithPrincipalsDto(role, rp.map(_.principalName))
  }

  @ReadOnlyTransactionalPermissionService
  override def readPrincipals(roleName: String): List[String] =
    rolePrincipalService.read(roleName).map(_.principalName)

  @ReadOnlyTransactionalPermissionService
  override def read(rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val pageable = PageRequest.of(page - 1, size, SortOrder.getSort(order, field))
    val resultPage = roleWithPrincipalsToDto(pageable, rolePrincipalService.read(rolePattern, pageable))
    PaginatedResponse[RoleWithPrincipalsDto](
      resultPage.getContent.asScala.toList,
      resultPage.getTotalElements,
      page,
      size,
      resultPage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def read(principal: String,
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val rolePrincipalsPage =
      rolePrincipalService.read(principal, rolePattern, PageRequest.of(page - 1, size, SortOrder.getSort(order, field)))
    PaginatedResponse[RoleWithPrincipalsDto](
      rolePrincipalsPage.getContent.asScala.toList,
      rolePrincipalsPage.getTotalElements,
      page,
      size,
      rolePrincipalsPage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def read(principals: List[String],
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val rolePrincipalsPage =
      rolePrincipalService.read(principals, rolePattern, PageRequest.of(page - 1, size, SortOrder.getSort(order, field)))
    PaginatedResponse[RoleWithPrincipalsDto](
      rolePrincipalsPage.getContent.asScala.toList,
      rolePrincipalsPage.getTotalElements,
      page,
      size,
      rolePrincipalsPage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(rolePattern: String): List[RoleWithPrincipalsDto] =
    rolePrincipalService.readByRolePattern(rolePattern)

  @ReadOnlyTransactionalPermissionService
  override def read(principal: String, rolePattern: String): List[RoleWithPrincipalsDto] =
    rolePrincipalService.read(principal, rolePattern)

  @ReadOnlyTransactionalPermissionService
  override def read(principals: List[String], rolePattern: String): List[RoleWithPrincipalsDto] =
    rolePrincipalService.read(principals, rolePattern)
}
