package ai.digital.deploy.permissions.client.local
import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithReferencedPermissionsDto}
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleWithReferencedPermissionsConverter._
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, ReferencedPermissionServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.ReferencedPermission
import ai.digital.deploy.permissions.model.view.RoleReferencedPermissionView
import ai.digital.deploy.permissions.service.ReferencedPermissionService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Service
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalReferencedPermissionServiceClient(
  @Autowired referencedPermissionService: ReferencedPermissionService
) extends ReferencedPermissionServiceClient {
  @ReadOnlyTransactionalPermissionService
  override def read(referenceId: UUID, roleName: String): List[String] =
    referencedPermissionService.read(referenceId, roleName)

  @ReadOnlyTransactionalPermissionService
  override def read(referenceId: UUID,
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithReferencedPermissionsDto] = {
    val pageable = PageRequest.of(page - 1, size, SortOrder.getSort(order, field))
    val resultPage = roleReferencedPermissionViewToDto(
      referenceId,
      pageable,
      referencedPermissionService.read(referenceId, rolePattern, pageable)
    )
    PaginatedResponse[RoleWithReferencedPermissionsDto](
      resultPage.toList.asScala.toList,
      resultPage.getTotalElements,
      page,
      size,
      resultPage.hasNext
    )
  }

  override def addPermissions(referenceId: UUID,
                              roleName: String,
                              permissions: List[String]
  ): RoleWithReferencedPermissionsDto = {
    val (role, rp) = referencedPermissionService.add(referenceId, roleName, permissions)
    RoleWithReferencedPermissionsDto(role, referenceId, rp)
  }

  override def removePermissions(referenceId: UUID,
                                 roleName: String,
                                 permissions: List[String]
  ): RoleWithReferencedPermissionsDto = {
    val (role, rp) = referencedPermissionService.remove(referenceId, roleName, permissions)
    RoleWithReferencedPermissionsDto(role, referenceId, rp)
  }

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRole(roleId: UUID, permissions: List[String]): Set[UUID] =
    referencedPermissionService.getReferencesForRole(roleId, permissions)

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRoles(roleId: List[UUID], permissions: List[String]): Set[UUID] =
    referencedPermissionService.getReferencesForRoles(roleId, permissions)

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRoleName(roleName: String, permissions: List[String]): Set[UUID] =
    referencedPermissionService.getReferencesForRoleName(roleName, permissions)

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForPrincipal(principalName: String, permissions: List[String]): Set[UUID] =
    referencedPermissionService.getReferencesForPrincipalName(principalName, permissions)

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(referenceId: UUID, rolePattern: String): List[RoleWithReferencedPermissionsDto] =
    referencedPermissionService
      .readForRolePattern(referenceId, rolePattern).groupBy(_.role).flatMap {
        case (role, permissions) =>
          val groupByReference = permissions.groupBy(_.reference)
          groupByReference.map {
            case (reference, referencePermissions) =>
              RoleWithReferencedPermissionsDto(role, reference, referencePermissions.map(_.permissionName))
          }
      }.toList
}
