package ai.digital.deploy.pendo.utils

import com.xebialabs.deployit.booter.local.TypePluginMappingManager
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.lang.StringEscapeUtils.escapeJava
import org.apache.commons.lang3.StringUtils
import org.apache.commons.text.CaseUtils

import java.util.Date

object PendoUtils extends Logging {

  private val PENDO_MAX_PROPERTY_LENGTH: Int = 1000

  implicit class PendoMap(val properties: Map[String, Any]) extends AnyVal {
    def toPendo: Map[String, Any] = {
      properties.view.mapValues {
        case s: String => if (s.length > PENDO_MAX_PROPERTY_LENGTH) s.substring(0, PENDO_MAX_PROPERTY_LENGTH) else s
        case date: Date => date
        case default => default
      }.toMap
    }
  }

  def generatePendoAccountId(accountData: Map[String, Object]): String = {
    val accountDataName = Option(accountData.getOrElse("name", "")) match {
      case Some(name) => name.asInstanceOf[String]
      case None => ""
    }
    val accountName = CaseUtils.toCamelCase(escapeJava(StringUtils.stripAccents(accountDataName))
      .replaceAll("[^a-zA-Z0-9]", " ").replaceAll("(\\d)([A-Za-z])", "$1 $2"), true)
    val accountEnvironment = DigestUtils.md5Hex(accountData.getOrElse("serverUrl", "").asInstanceOf[String]).substring(0, 6)
    val accountId = s"${accountName}_$accountEnvironment"
    accountId
  }

  def splitPropertyList(stringList: List[String]): Array[String] = {
    val (firstList, limiterOpt) = getSublistAndLimiter(stringList)

    limiterOpt match {
      case Some(limiter) =>
        val secondStringList = stringList.splitAt(limiter)._2
        val (secondList, _) = getSublistAndLimiter(secondStringList)
        Array(firstList, secondList)
      case None => Array(firstList, "")
    }
  }

  private def getSublistAndLimiter(stringList: List[String]) = {
    val sb = new StringBuffer
    var limitingIndex: Option[Int] = None // the first one that didn't make it in

    for (i <- stringList.indices) {
      val string = stringList(i)
      if (sb.length() + string.length + 2 < PENDO_MAX_PROPERTY_LENGTH) { // 2 because we're gonna remove the ","
        sb.append(string)
        sb.append(",")
      } else if (limitingIndex eq None) {
        limitingIndex = Some(i)
      }
    }

    sb.length() match {
      case l if l > 0 =>
        val sbWithoutFinalComma = sb.substring(0, l - 1)
        (sbWithoutFinalComma, limitingIndex)
      case l if l == 0 => ("", None)
    }
  }

  def getStepsMetadata(stepsMetadata: List[Map[String, String]]): Map[String, String] = {
    var applicationTypes = Map.empty[String, Int]
    var infrastructureTypes = Map.empty[String, Int]
    for(stepMetadata <- stepsMetadata) {
      for((key, value) <- stepMetadata) {
        if(key.startsWith("applicationType_")) {
          applicationTypes += applicationTypes.get(value).map(x => value -> (x + 1)).getOrElse(value -> 1)
        } else if(key.startsWith("infrastructureType")) {
          infrastructureTypes += infrastructureTypes.get(value).map(x => value -> (x + 1)).getOrElse(value -> 1)
        }
      }
    }
    Map(
      "applicationType" -> applicationTypes.map(_.productIterator.mkString("-")).mkString(","),
      "infrastructureType" -> infrastructureTypes.map(_.productIterator.mkString("-")).mkString(",")
    ) ++ createPluginsProperty(applicationTypes.keys)
  }

  private def getPluginData(url: String): Option[String] = {
    if (url != null) {
      val values: List[String] = url.split("/").filter((s: String) => s.contains(".xldp") || s.contains(".jar")).toList
      return values.headOption.map(getPluginType(url) + "_"+_.replaceAll("\\.xldp!?|\\.jar!?$", ""))
    }
    None
  }

  private def getPluginType(url: String): String = url match {
    case s if s.contains("xld-official") => "official"
    case s if s.contains("__local__") => "custom"
    case _ => ""
  }

  private def createPluginsProperty(applicationTypes: Iterable[String]): Map[String, String] = {
    val pluginsList =
      for {
        pluginUrl <- applicationTypes.map(getPluginURL)
        pluginData <- getPluginData(pluginUrl)
      } yield pluginData
    val propertyArray = splitPropertyList(pluginsList.toList)
    Map("plugins1" -> propertyArray(0), "plugins2" -> propertyArray(1))
  }

  private def getPluginURL(applicationType: String): String = {
    TypePluginMappingManager.getPluginUrl(applicationType)
  }
}
