package ai.digital.deploy.pendo

import ai.digital.deploy.pendo.actor.PendoDataPublisherInitializer
import ai.digital.deploy.settings.FeatureSettings
import ai.digital.deploy.settings.service.FeatureSettingsService
import com.xebialabs.analytics.pendo.PendoDataPublisher.{DisablePublishing, EnablePublishing}
import com.xebialabs.deployit.engine.spi.event.{AuditableDeployitEvent, CisCreatedEvent, CisUpdatedEvent, SpringInstantiatedListener, SystemStartedEvent}
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.plugin.api.reflect.Type
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.scheduling.annotation.Async
import org.springframework.stereotype.Component

import java.util.Optional
import java.util.concurrent.atomic.AtomicBoolean
import scala.jdk.OptionConverters._
import scala.util.{Failure, Success, Try}

@Component
class PendoSwitch(@Autowired featureSettingsService: FeatureSettingsService) extends SpringInstantiatedListener with Logging {

  val pendoEnabled: AtomicBoolean = new AtomicBoolean(false)
  val PENDO_TYPE: Type = Type.valueOf("settings.PendoAnalytics")

  def isEnabled: Boolean = pendoEnabled.get()

  @Subscribe
  def onPendoToggleEvent(event: PendoToggleEvent): Unit = togglePendoInternal(event.pendoEnabled)

  @Subscribe
  @Async
  def receiveSystemStart(event: SystemStartedEvent): Unit = {
    togglePendo(isPendoEnabled)
  }

  @Subscribe
  @Async
  def onCreate(event: CisCreatedEvent): Unit = {
    val pendoCi = event.getCis.stream().filter(ci => PENDO_TYPE.equals(ci.getType)).findAny()
    pendoCi.toScala match {
      case Some(settings) => togglePendo(settings.getProperty("enabled"))
      case _ =>
    }
  }

  @Subscribe
  @Async
  def onUpdate(event: CisUpdatedEvent): Unit = {
    val pendoCi = event.getCis.stream().filter(ci => PENDO_TYPE.equals(ci.getType)).findAny()
    pendoCi.toScala match {
      case Some(settings) => togglePendo(settings.getProperty("enabled"))
      case _ =>
    }
  }

  private def togglePendo(enabled: Boolean): Unit = {
    EventBusHolder.publish(PendoToggleEvent(enabled))
  }

  private def togglePendoInternal(enabled: Boolean): Unit = {
    val msg = if (enabled) EnablePublishing else DisablePublishing
    pendoEnabled.set(enabled)
    PendoDataPublisherInitializer.getInstance() ! msg
    logger.debug(s"Pendo analytics ${if (enabled) "enabled" else "disabled"}")
  }

  private def isPendoEnabled: Boolean = {
    Try(featureSettingsService.getFeatureSettings()) match {
      case Success(featureSettings) =>
        val pendoSettings: Optional[FeatureSettings] = featureSettings.stream().filter(featureSetting => PENDO_TYPE.equals(featureSetting.getType)).findAny()
        pendoSettings.toScala match {
          case Some(settings) => settings.getProperty("enabled")
          case _ => false
        }
      case Failure(exception) =>
        logger.warn(s"No pendo settings available, keeping Pendo analytics disabled. Reason: ${exception.getMessage}")
        false
    }
  }
}

case class PendoToggleEvent(pendoEnabled: Boolean) extends AuditableDeployitEvent("pendo", "pendo event")
