package ai.digital.deploy.pendo

import ai.digital.deploy.pendo.deployment.state.events.PendoDeploymentTaskData
import ai.digital.deploy.pendo.utils.DeploymentExtractor
import ai.digital.deploy.task.status.events.PendoDeploymentStateEvent
import com.xebialabs.analytics.pendo.{PendoEvent, PendoEventQueue}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import grizzled.slf4j.Logging
import jakarta.jms.TextMessage
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component

@Component
class PendoDeploymentStateConsumer(
                                       jacksonJmsMessageConverter: MessageConverter,
                                       @Autowired pendoSwitch: PendoSwitch, @Autowired eventQueue: PendoEventQueue
                                     ) extends Logging {
  @JmsListener(
    destination = "#{@pendoDeploymentStateQueueNameResolver.getPendoDeploymentStateQueueName()}",
    containerFactory = "xlJmsListenerContainerFactory"
  )
  def receiveTextMessage(message: TextMessage): Unit = {
    try {
      jacksonJmsMessageConverter.fromMessage(message) match {
        case event: PendoDeploymentStateEvent =>
          processEventData(event.currentState, event.taskData)
        case _ =>
          logger.error(s"Received unknown ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} [$message]")
      }
    } catch {
      case e: Exception => logger.warn("Exception occurred in processing message for " +
        "DeploymentPackageStatusConsumer Listener , message - ", message.getText, "with stacktrace", ExceptionUtils.getStackTrace(e))
    }
  }

  private def generatePendoEventData(currentState: TaskExecutionState, taskData: PendoDeploymentTaskData): Option[PendoEvent] = {
    currentState match {
      case TaskExecutionState.PENDING => DeploymentExtractor.deploymentCreatedData(taskData)
      case TaskExecutionState.QUEUED => DeploymentExtractor.deploymentStartedData(taskData)
      case TaskExecutionState.ABORTED => DeploymentExtractor.deploymentAbortedData(taskData)
      case TaskExecutionState.CANCELLED => DeploymentExtractor.deploymentCancelledData(taskData)
      case TaskExecutionState.FAILED => DeploymentExtractor.deploymentFailedData(taskData)
      case TaskExecutionState.EXECUTED => DeploymentExtractor.deploymentExecutedData(taskData)
      case _ => None
    }
  }

  private def processEventData(currentState: TaskExecutionState, taskData: PendoDeploymentTaskData): Unit = {
    if (pendoSwitch.isEnabled) {
      generatePendoEventData(currentState, taskData) match {
        case Some(extractedData) => eventQueue.add(extractedData)
        case None => logger.debug(s"There is no pendo data extractor for status $currentState, skipping")
      }
    }
  }
}
