package ai.digital.deploy.pendo.utils

import ai.digital.deploy.pendo.metrics.DataStatisticReports
import ai.digital.deploy.pendo.utils.PendoAccountMetadataConstants._
import com.xebialabs.deployit.support.report.{GeneralUsageInfo, ServerClusterInfo}
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.service.LicenseService
import grizzled.slf4j.Logging

import java.text.SimpleDateFormat
import java.util.Date
import scala.collection.mutable

object PendoAccountMetadataUtils extends Logging {


  def generatePendoAccountMetadata(dataStatisticReports: DataStatisticReports, licenseService: LicenseService,
                                   serverClusterInfo: ServerClusterInfo, generalUsageInfo: GeneralUsageInfo): mutable.Map[String, Any] = {
    val properties = mutable.Map[String, Any]()
    properties.put(REPORT_DATE, new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").format(new Date()))
    addDeploymentStatistics(dataStatisticReports, properties)
    addLicenseInfo(licenseService, properties)
    addInstallationMetrics(serverClusterInfo, generalUsageInfo, properties)
    addUsageMetrics(dataStatisticReports, properties)
    addControlTasksMetrics(dataStatisticReports, properties)
    addPluginMetrics(dataStatisticReports, properties)
    properties
  }

  private def addDeploymentStatistics(reports: DataStatisticReports, properties: mutable.Map[String, Any]): Unit = {
    val deploymentStatistics = reports.deploymentStatistics
    properties.put(STARTED_DEPLOYMENTS_LAST_YEAR_TOTAL, deploymentStatistics.totalDeploymentsInLastYear)
    properties.put(SUCCESSFUL_DEPLOYMENTS_LAST_YEAR_TOTAL, deploymentStatistics.successfulDeploymentsInLastYear)
    properties.put(RETRIED_DEPLOYMENTS_LAST_YEAR_TOTAL, deploymentStatistics.retriedDeploymentsInLastYear)
    properties.put(ROLLBACK_DEPLOYMENTS_LAST_YEAR_TOTAL, deploymentStatistics.rollbackDeploymentsInLastYear)
    properties.put(CANCELLED_DEPLOYMENTS_LAST_YEAR_TOTAL, deploymentStatistics.abortedDeploymentsInLastYear)
    properties.put(PENDING_DEPLOYMENTS_TOTAL, deploymentStatistics.totalPendingDeployment)
    properties.put(SCHEDULED_DEPLOYMENTS_TOTAL, deploymentStatistics.totalScheduledDeployment)
    properties.put(QUEUED_DEPLOYMENTS_TOTAL, deploymentStatistics.totalQueuedDeployment)
    properties.put(ACTIVE_DEPLOYMENTS_TOTAL, deploymentStatistics.activeDeployments)
    properties.put(DEPLOYMENTS_LAST_24HOURS_TOTAL, deploymentStatistics.totalDeploymentFor24hrs)
    properties.put(SUCCESSFUL_DEPLOYMENTS_LAST_24HOURS_TOTAL, deploymentStatistics.successfulDeploymentsIn24hrs)
    properties.put(RETRIED_DEPLOYMENTS_LAST_24HOURS_TOTAL, deploymentStatistics.retriedDeploymentsIn24hrs)
    properties.put(ROLLBACK_DEPLOYMENTS_LAST_24HOURS_TOTAL, deploymentStatistics.rollbackDeploymentsIn24hrs)
    properties.put(ABORTED_DEPLOYMENTS_LAST_24HOURS_TOTAL, deploymentStatistics.abortedDeploymentsIn24hrs)
    properties.put(DEPLOYMENTS_LAST_30DAYS_TOTAL, deploymentStatistics.totalDeploymentFor30days)
    properties.put(AVERAGE_DEPLOYMENT_TASKS_PER_DAY_PER_MONTH, deploymentStatistics.averageDeploymentPerDayPerMonth)
    properties.put(AVERAGE_DEPLOYMENT_TASKS_PER_DAY_PER_YEAR, deploymentStatistics.averageDeploymentPerDayPerYr)
  }

  private def addLicenseInfo(licenseService: LicenseService, properties: mutable.Map[String, Any]): Unit = {
    val licenseCiInUse = licenseService.getCounter().licensedCisInUse()
    val license = licenseService.getLicense
    properties.put(LICENSED_PRODUCT, license.getStringValue(LicenseProperty.PRODUCT))
    properties.put(LICENSED_TO, license.getStringValue(LicenseProperty.LICENSED_TO))
    properties.put(LICENSE_VERSION, license.getStringValue(LicenseProperty.LICENSE_VERSION))
    properties.put(LICENSE_EDITION, license.getStringValue(LicenseProperty.EDITION))
    properties.put(LICENSE_EXPIRES_AFTER, license.getStringValue(LicenseProperty.EXPIRES_AFTER))
    properties.put(LICENSE_PLUGINS, String.join(",", license.getListValue(LicenseProperty.LICENSED_PLUGINS)))
    // make sure keys are available irrespective of the license type
    properties.put(LICENSE_OVERTHERE_CI_COUNT, 0)
    properties.put(LICENSE_DEPLOYED_APPLICATION_CI_COUNT, 0)
    properties.put(INSTANCE_DEPLOYED_APPLICATION_CI_COUNT, 0)
    properties.put(INSTANCE_OVERTHERE_CI_COUNT, 0)
    properties.put(LICENSE_SATELLITE_CI_COUNT, 0)
    properties.put(INSTANCE_SATELLITE_CI_COUNT, 0)
    val cisList = List(HOST, SATELLITE, APPLICATION)
    licenseCiInUse.filter(ci => cisList.contains(ci.`type`.getName)).foreach(ciUse => {
      val allowedKey = LICENSE + ciUse.`type`.getName + CI_COUNT
      val actualKey = INSTANCE + ciUse.`type`.getName + CI_COUNT
      properties.put(allowedKey, ciUse.allowedAmount.toString)
      properties.put(actualKey, ciUse.actualAmount.toString)
    })
  }

  private def addInstallationMetrics(serverClusterInfo: ServerClusterInfo, generalUsageInfo: GeneralUsageInfo,
                                     properties: mutable.Map[String, Any]): Unit = {
    properties.put(DEPLOY_VERSION, generalUsageInfo.getDeployVersion())
    properties.put(VM_VERSION, generalUsageInfo.getVmName())
    properties.put(VM_ARGUMENTS, generalUsageInfo.getVmArguments())
    properties.put(OS, generalUsageInfo.getOsName())
    properties.put(OS_ARCHITECTURE, generalUsageInfo.getOsArchitecture())
    properties.put(OS_AVAILABLE_PROCESSORS, generalUsageInfo.getOsAvailableProcessors())
    properties.put(SYSTEM_NAME, generalUsageInfo.getSystemName())
    properties.put(MAX_MEMORY, generalUsageInfo.getMaxMemory())
    properties.put(USED_MEMORY, generalUsageInfo.getUsedMemory())
    properties.put(MAIN_DB_TYPE, generalUsageInfo.getDatabaseImplementation())
    properties.put(ARCHIVE_DB_TYPE, generalUsageInfo.getReportingDatabaseImplementation())
    properties.put(CLUSTER_MODE, serverClusterInfo.getMode())
    properties.put(CLUSTER, serverClusterInfo.isCluster)
    properties.put(NUMBER_OF_NODES, serverClusterInfo.getNumberOfNodes())
  }

  private def addUsageMetrics(dataStatisticReports: DataStatisticReports,
                              properties: mutable.Map[String, Any]): Unit = {
    val ciStatistics = dataStatisticReports.ci
    properties.put(APPLICATIONS_TOTAL, ciStatistics.totalApplication)
    properties.put(HOSTS_TOTAL, ciStatistics.totalHosts)
    properties.put(SATELLITES_TOTAL, ciStatistics.totalSatellites)
    properties.put(WORKERS_TOTAL, ciStatistics.totalWorker)
    val userStatistics = dataStatisticReports.usersAndRoleStatistics
    properties.put(ACTIVE_USERS_TOTAL, userStatistics.totalActiveUsers)
    properties.put(GLOBAL_ROLES_TOTAL, userStatistics.totalGlobalRoles)
    val folderStatistics = dataStatisticReports.folderStatistics
    properties.put(APPLICATION_FOLDERS_TOTAL, folderStatistics.applicationFolderCount)
    properties.put(ENVIRONMENT_FOLDERS_TOTAL, folderStatistics.environmentFolderCount)
    properties.put(INFRASTRUCTURE_FOLDERS_TOTAL, folderStatistics.infraFolderCount)
    val placeholderStatistics = dataStatisticReports.placeholderStatistics
    properties.put(PLACEHOLDERS_TOTAL, placeholderStatistics.placeholdersCount)
    //add external internal users with default value 0
    properties.put(USERS_TOTAL, 0)
    properties.put(ENABLED_USERS_TOTAL, 0)
    properties.put(INTERNAL_USERS_TOTAL, 0)
    properties.put(EXTERNAL_USERS_TOTAL, 0)
  }

  private def addControlTasksMetrics(dataStatisticReports: DataStatisticReports,
                                     properties: mutable.Map[String, Any]): Unit = {
    val controlTaskStatistics = dataStatisticReports.controlTaskUsageStatistics
    properties.put(PENDING_CONTROL_TASKS_TOTAL, controlTaskStatistics.totalPendingTasks)
    properties.put(SCHEDULED_CONTROL_TASKS_TOTAL, controlTaskStatistics.totalScheduledTasks)
    properties.put(QUEUED_CONTROL_TASKS_TOTAL, controlTaskStatistics.totalQueuedTasks)
    properties.put(ACTIVE_CONTROL_TASKS_TOTAL, controlTaskStatistics.totalActiveTasks)
    properties.put(CONTROL_TASKS_LAST_24HOURS_TOTAL, controlTaskStatistics.totalTaskFor24hrs)
    properties.put(CONTROL_TASKS_LAST_30DAYS_TOTAL, controlTaskStatistics.totalTaskFor30days)
    properties.put(SUCCESSFUL_CONTROL_TASKS_LAST_24HOURS, controlTaskStatistics.totalSuccessfulTasksIn24Hours)
    properties.put(ABORTED_CONTROL_TASKS_LAST_24HOURS, controlTaskStatistics.totalAbortedTasksIn24Hours)
    properties.put(AVERAGE_CONTROL_TASKS_PER_DAY_PER_MONTH, controlTaskStatistics.averageTaskPerDayPerMonth)
    properties.put(AVERAGE_CONTROL_TASKS_PER_DAY_PER_YEAR, controlTaskStatistics.averageTaskPerDayPerYr)
  }

  private def addPluginMetrics(dataStatisticReports: DataStatisticReports, properties: mutable.Map[String, Any])= {
    val pluginStatistics = dataStatisticReports.pluginStatistics
    properties.put(PLUGIN_DATA_BY_DEPLOYMENT_LAST_180_DAYS, pluginStatistics.pluginsDataByDeployment)
  }

}
