package ai.digital.deploy.pendo

import ai.digital.deploy.pendo.utils.DeploymentExtractor
import com.xebialabs.analytics.pendo.{PendoEvent, PendoEventQueue}
import com.xebialabs.deployit.core.events.DeploymentStateEvent
import com.xebialabs.deployit.engine.api.execution.{TaskExecutionState, TaskWithSteps}
import com.xebialabs.deployit.engine.spi.event.SpringInstantiatedListener
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.scheduling.annotation.Async
import org.springframework.stereotype.Component

@Component
class PendoRelevantEventHandler(@Autowired pendoSwitch: PendoSwitch, @Autowired eventQueue: PendoEventQueue) extends SpringInstantiatedListener with Logging {

  @Async
  @Subscribe
  def onDeploymentStateChange(event: DeploymentStateEvent): Unit = {
    processEventData(event.currentState, event.task)
  }

  private def generatePendoEventData(currentState: TaskExecutionState, task: TaskWithSteps): Option[PendoEvent] = {
    currentState match {
      case TaskExecutionState.PENDING => DeploymentExtractor.deploymentCreatedData(task)
      case TaskExecutionState.QUEUED => DeploymentExtractor.deploymentStartedData(task)
      case TaskExecutionState.ABORTED => DeploymentExtractor.deploymentAbortedData(task)
      case TaskExecutionState.CANCELLED => DeploymentExtractor.deploymentCancelledData(task)
      case TaskExecutionState.FAILED => DeploymentExtractor.deploymentFailedData(task)
      case TaskExecutionState.EXECUTED => DeploymentExtractor.deploymentExecutedData(task)
      case _ => None
    }
  }

  private def processEventData(currentState: TaskExecutionState, task: TaskWithSteps): Unit = {
    if (pendoSwitch.isEnabled) {
      generatePendoEventData(currentState, task) match {
        case Some(extractedData) => eventQueue.add(extractedData)
        case None => logger.debug(s"There is no pendo data extractor for status $currentState, skipping")
      }
    }
  }
}
