package ai.digital.deploy.pendo.metrics

import ai.digital.deploy.pendo.PendoConstants._
import ai.digital.deploy.pendo.PendoSwitch
import com.xebialabs.analytics.pendo.{PendoEvent, PendoEventQueue}
import com.xebialabs.deployit.support.report.{CiStatistics, ControlTaskStatistics, DeploymentStatistics, FolderStatistics, GeneralUsageInfo, PlaceholderStatistics, ServerClusterInfo, UserAndRoleStatistics}
import com.xebialabs.deployit.support.rest.SystemInformationService
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.service.LicenseService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.time.Instant
import scala.collection.mutable
import scala.jdk.CollectionConverters.MapHasAsScala

case class DataStatisticReports(ci: CiStatistics, deploymentStatistics: DeploymentStatistics, folderStatistics: FolderStatistics,
                                controlTaskUsageStatistics: ControlTaskStatistics, usersAndRoleStatistics: UserAndRoleStatistics,
                                placeholderStatistics: PlaceholderStatistics)

@Component
class XldPendoMetricsService(eventQueue: PendoEventQueue, pendoSwitch: PendoSwitch) extends Logging {

  @Autowired var licenseService: LicenseService = _
  @Autowired var systemInformationService: SystemInformationService = _

  def processMetrics(): Unit = {
    if (pendoSwitch.isEnabled) {
      try {
        val systemInfo = systemInformationService.generateJsonReport(Seq("cluster", "general", "data")).asScala.toMap
        val clusterStats = getStatisticsAs[ServerClusterInfo]("cluster", systemInfo)
        val generalStats = getStatisticsAs[GeneralUsageInfo]("general", systemInfo)
        val usageStatsAsMap = getStatisticsAs[java.util.Map[String, Any]]("data", systemInfo).asScala.toMap
        val usageStats = extractUsageStatistics(usageStatsAsMap)

        eventQueue.add(generateLicenseMetricsEvent())
        eventQueue.add(generateUsageMetricsEvent(usageStats.ci,
          usageStats.deploymentStatistics, usageStats.folderStatistics, usageStats.controlTaskUsageStatistics,
          usageStats.usersAndRoleStatistics, usageStats.placeholderStatistics))
        eventQueue.add(generateInstallationMetricsEvent(generalStats, clusterStats))
      } catch {
        case t: Throwable => logger.warn(s"Unable to send metrics to Pendo, error: ${t.getMessage}")
      }
    }
  }

  def extractUsageStatistics(dataStatisticsReport: Map[String, Any]): DataStatisticReports = {
    DataStatisticReports(getStatisticsAs[CiStatistics]("ci", dataStatisticsReport),
      getStatisticsAs[DeploymentStatistics]("deployments", dataStatisticsReport),
      getStatisticsAs[FolderStatistics]("folders", dataStatisticsReport),
      getStatisticsAs[ControlTaskStatistics]("tasks", dataStatisticsReport),
      getStatisticsAs[UserAndRoleStatistics]("users", dataStatisticsReport),
      getStatisticsAs[PlaceholderStatistics]("placeholders", dataStatisticsReport))
  }

  def getStatisticsAs[T](key: String, statMap: Map[String, Any]): T = Option(statMap.get(key)) match {
    case Some(stats) => stats.getOrElse(null).asInstanceOf[T]
    case None => throw new NoSuchElementException(s"Unable to find statistics data '${key}' in System Information report")
  }


   def generateLicenseMetricsEvent(): PendoEvent = {
    val licenseCiInUse = licenseService.getCounter.licensedCisInUse
    val license = licenseService.getLicense
    var properties = mutable.Map[String, Any]()
    properties ++= List(PRODUCT_NAME -> DEPLOY,
      LICENSE_EXPIRATION_DATE -> license.getStringValue(LicenseProperty.EXPIRES_AFTER),
      CUSTOMER_NAME -> license.getStringValue(LicenseProperty.LICENSED_TO),
      LICENSE_VERSION -> license.getStringValue(LicenseProperty.LICENSE_VERSION),
      LICENSE_PLUGINS -> String.join(",", license.getListValue(LicenseProperty.LICENSED_PLUGINS)),
      LICENSE_EDITION -> license.getStringValue(LicenseProperty.EDITION),
      LICENSE_SATELLITE -> "",
      INSTANCE_SATELLITE -> "",
      LICENSE_HOST -> "",
      INSTANCE_HOST -> "",
      LICENSE_APPLICATION -> "",
      INSTANCE_APPLICATION -> ""
    )
    for (ciUse <- licenseCiInUse) {
      val allowedKey = "license" + ciUse.`type`.getName
      val actualKey = "instance" + ciUse.`type`.getName
      if (properties.contains(allowedKey) && properties.contains(actualKey)) {
        properties.put(allowedKey, ciUse.allowedAmount.toString)
        properties.put(actualKey, ciUse.actualAmount.toString)
      }
    }
    PendoEvent(LICENSE_METRICS, properties.toMap, Instant.now.toEpochMilli)
  }

  def generateUsageMetricsEvent(ci: CiStatistics, deployments: DeploymentStatistics, folders: FolderStatistics,
                                controlTasks: ControlTaskStatistics, users: UserAndRoleStatistics,
                                placeholders: PlaceholderStatistics): PendoEvent = {
    val properties = Map(TOTAL_APPLICATIONS -> ci.totalApplication.toString, TOTAL_HOSTS -> ci.totalHosts.toString,
      TOTAL_SATELLITES -> ci.totalSatellites.toString, TOTAL_WORKERS -> ci.totalWorker.toString,
      TOTAL_DEPLOYMENTS_24_HRS -> deployments.totalDeploymentFor24hrs, TOTAL_DEPLOYMENTS_30_DAYS -> deployments.totalDeploymentFor30days,
      TOTAL_PENDING_DEPLOYMENTS -> deployments.totalPendingDeployment, TOTAL_SCHEDULED_DEPLOYMENTS -> deployments.totalScheduledDeployment,
      TOTAL_QUEUED_DEPLOYMENTS -> deployments.totalQueuedDeployment, TOTAL_ACTIVE_DEPLOYMENTS -> deployments.activeDeployments,
      TOTAL_TASKS_24_HRS -> controlTasks.totalTaskFor24hrs, TOTAL_TASKS_30_DAYS -> controlTasks.totalTaskFor30days,
      TOTAL_ACTIVE_TASKS -> controlTasks.totalActiveTasks, TOTAL_SCHEDULED_TASKS -> controlTasks.totalScheduledTasks,
      TOTAL_PENDING_TASKS -> controlTasks.totalPendingTasks, TOTAL_QUEUED_TASKS -> controlTasks.totalQueuedTasks,
      TOTAL_APPLICATION_FOLDERS -> folders.applicationFolderCount, TOTAL_ENVIRONMENT_FOLDERS -> folders.environmentFolderCount,
      TOTAL_INFRASTRUCTURE_FOLDERS -> folders.infraFolderCount, TOTAL_ACTIVE_USERS -> users.totalActiveUsers,
      TOTAL_PLACEHOLDERS -> placeholders.placeholdersCount)
    PendoEvent(USAGE_METRICS, properties, Instant.now.toEpochMilli)
  }

  def generateInstallationMetricsEvent(general: GeneralUsageInfo, cluster: ServerClusterInfo): PendoEvent = {
    val properties = Map(DEPLOY_VERSION -> general.deployVersion, MAX_MEMORY -> general.maxMemory / 1000,
      MAIN_DB_TYPE -> general.databaseImplementation, REPORTING_DB_TYPE -> general.reportingDatabaseImplementation,
      CLUSTER_MODE -> cluster.mode, NUMBER_OF_NODES -> cluster.numberOfNodes.toString,
      VM_VERSION -> general.vmName, OS_NAME -> general.osName, USED_MEMORY -> general.usedMemory / 1000)
    PendoEvent(INSTALLATION_METRICS, properties, Instant.now.toEpochMilli)
  }
}
