package ai.digital.deploy.pendo

import ai.digital.configuration.central.deploy.ServerSideProperties
import ai.digital.deploy.pendo.actor.{PendoActorSystem, PendoDataPublisherInitializer}
import ai.digital.deploy.pendo.utils.PendoUtils.generatePendoAccountId
import akka.actor.ActorSystem
import com.xebialabs.analytics.pendo.{PendoEventQueue, PendoSender}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.core.service.PendoMetadataService
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.service.LicenseService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration}

import java.net.InetAddress

@Configuration
class XldPendoConfiguration extends Logging {

  @Autowired
  var serverConfiguration: ServerConfiguration = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var serverSideConfiguration: ServerSideProperties = _

  @Autowired
  var pendoMetadataService: PendoMetadataService = _

  @Bean
  def pendoEventQueue(): PendoEventQueue = new PendoEventQueue(serverSideConfiguration.analytics.pendoQueueSize)

  @Bean
  def pendoSender(): PendoSender = {
    implicit val actorSystem: ActorSystem = PendoActorSystem.actorSystem
    val accountInfo = Map("serverUrl" -> serverConfiguration.getServerUrl, "name" -> licenseService.getLicense.getStringValue(LicenseProperty.LICENSED_TO))
    val host = InetAddress.getLocalHost.getHostName
    val accountId = getPendoAccountId(accountInfo)
    val visitorId = s"$accountId/$host"
    new PendoSender(accountId, visitorId, PendoConstants.TRACKED_EVENT_INTEGRATION_KEY)
  }

  @Bean
  def pendoDataPublisherInitializer(queue: PendoEventQueue, sender: PendoSender): PendoDataPublisherInitializer = {
    new PendoDataPublisherInitializer(queue, sender)
  }

  private def getPendoAccountId(accountInfo: Map[String, Object]): String = {
    var accountId: String = null
    try {
      accountId = pendoMetadataService.getPendoAccountId()
    } catch {
      case e: Exception => logger.error("Exception occurred while fetching pendo account id", e)
    }
    if (accountId == null) {
      accountId = generatePendoAccountId(accountInfo)
      try {
        pendoMetadataService.savePendoAccountId(accountId)
      } catch {
        case e: Exception => logger.error("Exception occurred while saving pendo account id", e)
      }
    }
    accountId
  }
}
