package ai.digital.deploy.notification.service

import ai.digital.deploy.notification.events.UserTokenAboutToExpireEvent
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.repository.{ConfigurationRepository, RepositoryService}
import com.xebialabs.deployit.security.service.{UserProfileService, UserTokenService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ai.digital.deploy.notification.email.MultipartEmailSender
import com.xebialabs.deployit.service.dependency.RepositoryServiceAware
import grizzled.slf4j.Logging
import ai.digital.deploy.notification.mail.PatSmtpServer

import java.time.temporal.ChronoUnit
import java.time.{Instant, LocalDate, ZoneId}
import java.util.Date
import scala.jdk.CollectionConverters.CollectionHasAsScala

/**
 * Service for sending email notifications on Personal Access Token expiration.
 * NOTE: This service directly accesses the ConfigurationRepository to retrieve settings.
 * Any changes to keys or structure in General Settings must be reflected in this class.
 */
@Component
class PATEmailNotificationService @Autowired()(userProfileService: UserProfileService,
                                               userTokenService: UserTokenService,
                                               multipartEmailSender: MultipartEmailSender,
                                               configurationRepository: ConfigurationRepository,
                                               override val repositoryService: RepositoryService) extends Logging  with RepositoryServiceAware {

  private lazy val serverUrl: String = ServerConfiguration.getInstance().getServerUrl

  def sendNotification(event: UserTokenAboutToExpireEvent): Unit = {
     logger.debug(s"PAT :: Sending PAT notification for user `${event.username}` with token `${event.userToken.getTokenNote}.")
    val settings = getGeneralSettings()
    val mailServerRefOpt = settings.get("pat_email_notification_smtp_server_ci_ref")
    val logoFileOpt = settings.get("logo_file")

    if (mailServerRefOpt.isEmpty) {
      logger.warn(s"PAT :: Email server not configured in general settings for token ${event.userToken.getTokenNote} of user ${event.username}.Skipped PAT notification.")
      return
    }

    val mailServerOpt = mailServerRefOpt.flatMap(getMailServer(_, event.userToken.getCiId))
    val emailOpt = getUserEmail(event.username)

    for {
      mailServer <- mailServerOpt
      email <- emailOpt
    } sendExpiryEmail(mailServer, email, event, logoFileOpt)
  }

  private def sendExpiryEmail(
                               mailServer: PatSmtpServer,
                               email: String,
                               event: UserTokenAboutToExpireEvent,
                               logoFileOpt: Option[String]): Unit = {
    val expiryDate = event.userToken.getExpiryDate
    val hasCustomLogo = logoFileOpt.exists(_.nonEmpty)
    val customLogoParam = logoFileOpt.filter(_.nonEmpty).map(logo => ("customLogo", logo))

    try {
      multipartEmailSender.sendCustomEmail(
        mailServer = mailServer, // SMTP server to use
        from = mailServer.getFromAddress, // Sender address
        to = List(email), // Recipient(s)
        subject = "[Deploy] Your personal access token is about to expire", // Subject
        templateVariables = Map(
          "tokenNote" -> event.userToken.getTokenNote,
          "tokenExpiryDate" -> formatExpiryDate(expiryDate),
          "accessTokensUrl" -> s"$serverUrl#/personalsettings/tokens",
          "url" -> serverUrl,
          "isCustomLogoSettingsDefined" -> hasCustomLogo
        ), // Variables for template
        baseTemplatePath = Some("templates/email_base_template.ftl"), // Required base template
        contentTemplate = Some("email_content_access_token.ftl"), // content template
        textContent = None, // No plain text content
        inlineImages = Seq("deploy-logo-white" -> "images/deploy-logo.png"), // Inline images
        customLogo = customLogoParam // custom logo
      )
      logger.debug(s"PAT :: Email notification sent successfully for expiring PAT token ${event.userToken.getTokenNote} for user: ${event.username}.")
    } catch {
      case ex: Exception =>
        logger.error(s"PAT :: Failed to send email for token ${event.userToken.getTokenNote} for user: ${event.username}. Exception: ${ex.getMessage}", ex)
        userTokenService.updateTokenExpiredNotified(event.userToken.getCiId, java.sql.Timestamp.from(Instant.now()), isNotified = false)
    }
  }

  private def getUserEmail(username: String): Option[String] = {
    try {
      val userEmailOpt = Option(userProfileService.findOne(username)).flatMap { profile =>
        Option(profile.email)
      }
      userEmailOpt match {
        case Some(email) if email.trim.nonEmpty => Some(email)
        case _ =>
          logger.warn(s"PAT :: No email configured for user $username. Skipped PAT notification.")
          None
      }
    } catch {
      case _ : com.xebialabs.deployit.exception.NotFoundException =>
        logger.warn(s"PAT :: No user profile found for user: $username. Skipped PAT notification.")
        None
    }
  }

  private def getMailServer(smtpId: String, tokenId: Int): Option[PatSmtpServer] = {
    try {
      repositoryService.listEntities[PatSmtpServer](typedSearchParameters[PatSmtpServer]).asScala
        .find(_.get$internalId() == smtpId.toInt)
        .orElse {
          logger.warn(s"PAT :: SMTP server with ID: $smtpId not found. Skipped PAT notification.")
          None
        }
    } catch {
      case _: com.xebialabs.deployit.exception.NotFoundException =>
        logger.warn(s"PAT :: SMTP server with ID: $smtpId not found. Skipped PAT notification.")
        // Setting isNotified as false when SMTP server is not found
        userTokenService.updateTokenExpiredNotified(tokenId, java.sql.Timestamp.from(Instant.now()), isNotified = false)
        None
    }
  }

  private def getGeneralSettings(): Map[String, String] = {
    configurationRepository.getConfigurationByKey
      .filter(conf => conf.key.endsWith("logo_file") || conf.key.endsWith("pat_email_notification_smtp_server_ci_ref"))
      .map(conf => conf.key.split("\\.").last -> conf.value)
      .toMap
  }

  private def formatExpiryDate(expiryDate: Date): String = {
    val today = LocalDate.now(ZoneId.of("UTC"))
    val expiry = expiryDate.toInstant.atZone(ZoneId.of("UTC")).toLocalDate
    val daysBetween = ChronoUnit.DAYS.between(today, expiry)
    if (daysBetween <= 0) {
      "today"
    } else {
      s"in $daysBetween day(s)"
    }
  }
}
