package ai.digital.deploy.notification

import grizzled.slf4j.Logging
import jakarta.jms.TextMessage
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component
import ai.digital.deploy.notification.events.UserTokenAboutToExpireEvent
import ai.digital.deploy.notification.service.PATEmailNotificationService

@Component
class DeployNotificationConsumer(@Autowired jacksonJmsMessageConverter: MessageConverter,
                                 @Autowired notificationService: PATEmailNotificationService) extends Logging {
  @JmsListener(
    destination = "#{@deployNotificationQueueNameResolver.getDeployNotificationQueueName()}",
    containerFactory = "xlJmsListenerContainerFactory"
  )
  def receiveTextMessage(message: TextMessage): Unit = {
    try {
      jacksonJmsMessageConverter.fromMessage(message) match {
        case event: UserTokenAboutToExpireEvent =>
          processUserTokenExpireNotificationEvent(event)
        case _ =>
          logger.error(s"Received unknown ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} [$message]")
      }
    } catch {
      case e: Exception => logger.warn("Exception occurred in processing message for " +
        "DeployNotificationConsumer Listener , message - ", message.getText, "with stacktrace", ExceptionUtils.getStackTrace(e))
    }
  }

  private def processUserTokenExpireNotificationEvent(event: UserTokenAboutToExpireEvent): Unit = {
    logger.trace(s"PAT :: Processing UserTokenAboutToExpireEvent for user: ${event.username}, " +
      s"token expiration time: ${event.userToken.getExpiryDate}, UserToken sent: ${event.userToken}")
    notificationService.sendNotification(event)
  }
}
