package com.xebialabs.xldeploy.jms.factory

import com.xebialabs.deployit.core.config.{ExternalConfig, InProcessConfig}
import com.xebialabs.xldeploy.jms.adapter.{UnsupportedDriverException, UnsupportedJmsConfigurationException}
import org.springframework.jms.connection.UserCredentialsConnectionFactoryAdapter
import org.springframework.stereotype.Component

import java.lang.reflect.Method
import javax.jms.ConnectionFactory

@Component
class DefaultJmsConnectionFactory extends BaseJmsConnectionFactory() {

  override protected def factory(): ConnectionFactory = {
    val factory = taskQueueBrokerConfig match {
      case cfg: ExternalConfig if !inProcessTaskEngine =>
        val baseCF = Class
          .forName(cfg.jmsDriverClassname)
          .asSubclass(classOf[ConnectionFactory])
          .getDeclaredConstructor()
          .newInstance()
        val cfWithAddress = setBrokerUrl(baseCF, cfg.url)
        val authenticatedCF =
          setUserCredentials(cfWithAddress, cfg.username, cfg.password)
        authenticatedCF
      case _: InProcessConfig if inProcessTaskEngine =>
        EmbeddedBrokerConnectionFactory.initialize()
      case _ =>
        throw UnsupportedJmsConfigurationException(
          "Unsupported configuration: deploy.task.in-process-worker should be set to 'false' to use external broker."
        )
    }
    factory
  }

  private def setBrokerUrl(cf: ConnectionFactory,
                           url: String): ConnectionFactory = {
    val method = findUrlSetter(cf, Seq("setUri", "setBrokerURL"))
    method match {
      case Some(mthd) => mthd.invoke(cf, url)
      case _ =>
        throw UnsupportedDriverException(
          "Cannot set JMS broker URL. Driver does not provide known mechanics for broker URL configuration."
        )
    }
    cf
  }

  private def findUrlSetter(cf: ConnectionFactory,
                            methodNames: Seq[String]): Option[Method] = {
    val options = methodNames.foldLeft[Seq[Method]](Seq.empty) {
      (acc: Seq[Method], name: String) =>
        try {
          val method = cf.getClass.getMethod(name, classOf[String])
          acc ++ Seq(method)
        } catch {
          case e: SecurityException =>
            logger.error(e)
            throw UnsupportedJmsConfigurationException(e.getMessage)
          case _: NoSuchMethodException =>
            logger.debug(s"JMS driver does not support setting broker URL using [$name].")
            acc
        }
    }
    options.headOption
  }

  private def setUserCredentials(cf: ConnectionFactory,
                                 username: String,
                                 password: String): ConnectionFactory = {
    val tcf = new UserCredentialsConnectionFactoryAdapter()
    tcf.setUsername(username)
    tcf.setPassword(password)
    tcf.setTargetConnectionFactory(cf)
    tcf
  }
}
