package com.xebialabs.xldeploy.jms.adapter

import ai.digital.configuration.central.deploy.TaskerSystemProperties
import ai.digital.configuration.central.deploy.converter.HoconDurationConverter
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.boot.autoconfigure.jms.artemis.ArtemisAutoConfiguration
import org.springframework.context.annotation._
import org.springframework.jms.annotation.EnableJms
import org.springframework.jms.config.{DefaultJmsListenerContainerFactory, JmsListenerContainerFactory}
import org.springframework.jms.core.JmsTemplate
import org.springframework.jms.listener.DefaultMessageListenerContainer
import org.springframework.jms.support.converter.{MappingJackson2MessageConverter, MessageConverter, MessageType}
import org.springframework.jms.support.destination.DestinationResolver
import org.springframework.util.ErrorHandler
import org.springframework.util.backoff.FixedBackOff

import java.util.concurrent.TimeUnit
import javax.jms.Session
import scala.language.reflectiveCalls

@Configuration
@Import(Array(
  classOf[ArtemisAutoConfiguration],
  classOf[TaskerSystemProperties]
))
@EnableJms
@ComponentScan(basePackages = Array("com.xebialabs.xldeploy.jms"))
class DefaultQueueJmsBean(@Autowired jmsConnectionFactoryProvider: JmsConnectionFactoryProvider) extends Logging {

  @Autowired
  var taskerSystemProperties: TaskerSystemProperties = _

  @Value("${deploy.server.events.cis-changed-queue-name:xld-cis-changed}")
  var cisChangedQueueName: String = _

  @Value("${deploy.server.events.deployment-package-status-queue-name:xld-deployment-package-status}")
  var deploymentPackageStatusQueueName: String = _

  @Value("${deploy.server.events.time-to-live:5 minutes}")
  var serverEventsTimeToLive: String = _

  @Value("${deploy.task.events.task-path-status.queue.time-to-live:5 minutes}")
  var taskPathStatusQueueTimeToLive: String = _

  @Value("${deploy.task.events.task-step-log.queue.name:xld-task-step-log}")
  var taskStepLogQueueName: String = _

  @Value("${deploy.task.events.task-step-log.queue.time-to-live:5 minutes}")
  var taskStepLogQueueTimeToLive: String = _

  @Bean
  def mainJmsTemplate(@Autowired destinationResolver: DestinationResolver): JmsTemplate = createJsmTemplate(destinationResolver)

  @Bean
  def notificationJmsTemplate(@Autowired destinationResolver: DestinationResolver): JmsTemplate = {
    val jmsMessagingTemplate = createJsmTemplate(destinationResolver)
    jmsMessagingTemplate.setTimeToLive(HoconDurationConverter.convert(serverEventsTimeToLive, TimeUnit.MILLISECONDS))
    jmsMessagingTemplate
  }

  @Bean
  def taskPathStatusJmsTemplate(@Autowired destinationResolver: DestinationResolver): JmsTemplate = {
    val jmsMessagingTemplate = createJsmTemplate(destinationResolver)
    jmsMessagingTemplate.setTimeToLive(HoconDurationConverter.convert(taskPathStatusQueueTimeToLive, TimeUnit.MILLISECONDS))
    jmsMessagingTemplate
  }

  @Bean
  def taskStepLogJmsTemplate(@Autowired destinationResolver: DestinationResolver): JmsTemplate = {
    val jmsMessagingTemplate = createJsmTemplate(destinationResolver)
    jmsMessagingTemplate.setTimeToLive(HoconDurationConverter.convert(taskStepLogQueueTimeToLive, TimeUnit.MILLISECONDS))
    jmsMessagingTemplate
  }

  private def createJsmTemplate(destinationResolver: DestinationResolver): JmsTemplate = {
    val jmsMessagingTemplate = new JmsTemplate(jmsConnectionFactoryProvider.cachingConnectionFactory())
    jmsMessagingTemplate.setMessageConverter(jacksonJmsMessageConverter)
    jmsMessagingTemplate.setDestinationResolver(destinationResolver)
    jmsMessagingTemplate
  }

  @Bean
  def defaultJmsErrorHandler: ErrorHandler = {
    (t: Throwable) => error("Error occurred while processing message from a queue", t)
  }

  @ConditionalOnProperty(prefix = "spring.artemis.embedded", name = Array("enabled"), havingValue = "true")
  @DependsOn(Array("embeddedActiveMq"))
  @Bean(Array("xlJmsListenerContainerFactory"))
  def jmsListenerContainerFactoryWithEmbeddedMq(@Autowired defaultJmsErrorHandler: ErrorHandler,
                                                @Autowired destinationResolver: DestinationResolver
                                               ): JmsListenerContainerFactory[DefaultMessageListenerContainer] =
    jmsListenerContainerFactory(defaultJmsErrorHandler, destinationResolver)

  @ConditionalOnProperty(prefix = "spring.artemis.embedded", name = Array("enabled"), havingValue = "false", matchIfMissing = true)
  @Bean(Array("xlJmsListenerContainerFactory"))
  def jmsListenerContainerFactoryWithStandaloneMq(@Autowired defaultJmsErrorHandler: ErrorHandler,
                                                  @Autowired destinationResolver: DestinationResolver
                                                 ): JmsListenerContainerFactory[DefaultMessageListenerContainer] =
    jmsListenerContainerFactory(defaultJmsErrorHandler, destinationResolver)

  @Bean
  def archiveQueueNameResolver(): ArchiveQueueNameResolver = new ArchiveQueueNameResolver {
    override def getQueueName: String = taskerSystemProperties.queue.archiveQueueName
  }

  @Bean
  def ciQueueNameResolver(): CiQueueNameResolver = new CiQueueNameResolver {
    override def getCisChangedQueueName: String = cisChangedQueueName
  }

  @Bean
  def taskPathStatusQueueNameResolver(): TaskPathStatusQueueNameResolver = new TaskPathStatusQueueNameResolver {
    override def getTaskPathStatusQueueName: String = taskerSystemProperties.events.taskPathStatus.queue.name
  }

  @Bean
  def deploymentPackageStatusQueueNameResolver(): DeploymentPackageStatusQueueNameResolver = new DeploymentPackageStatusQueueNameResolver {
    override def getDeploymentPackageStatusQueueName: String = deploymentPackageStatusQueueName
  }

  @Bean
  def taskPathStatusQueueFlushResolver(): TaskPathStatusQueueFlushResolver = new TaskPathStatusQueueFlushResolver {
    override def getEventFlushPeriodInMillis: Long = taskerSystemProperties.events.taskPathStatus.queue.getEventFlushPeriodInMillis
  }

  @Bean
  def taskStepLogQueueFlushResolver(): TaskStepLogQueueFlushResolver = new TaskStepLogQueueFlushResolver {
    override def getEventFlushPeriodInMillis: Long = taskerSystemProperties.events.taskStepLog.queue.getEventFlushPeriodInMillis
  }

  @Bean
  def taskStepLogQueueNameResolver(): TaskStepLogQueueNameResolver = new TaskStepLogQueueNameResolver {
    override def getTaskStepLogQueueName: String = taskStepLogQueueName
  }

  @Bean
  def destinationResolver(@Autowired archiveQueueNameResolver: ArchiveQueueNameResolver,
                          @Autowired taskPathStatusQueueNameResolver: TaskPathStatusQueueNameResolver,
                          @Autowired taskStepLogQueueNameResolver: TaskStepLogQueueNameResolver,
                          @Autowired deploymentPackageStatusQueueNameResolver: DeploymentPackageStatusQueueNameResolver): DestinationResolver =
    (session: Session, destinationName: String, _: Boolean) => {
      if (destinationName.equals(taskerSystemProperties.queue.name)) {
        session.createQueue(destinationName)
      }
      else if (destinationName.equals(archiveQueueNameResolver.getQueueName)) {
        session.createQueue(archiveQueueNameResolver.getQueueName)
      }
      else if (destinationName.equals(taskPathStatusQueueNameResolver.getTaskPathStatusQueueName)) {
        session.createQueue(taskPathStatusQueueNameResolver.getTaskPathStatusQueueName)
      }
      else if (destinationName.equals(taskStepLogQueueNameResolver.getTaskStepLogQueueName)) {
        session.createQueue(taskStepLogQueueNameResolver.getTaskStepLogQueueName)
      }
      else if (destinationName.equals(deploymentPackageStatusQueueNameResolver.getDeploymentPackageStatusQueueName)) {
        session.createQueue(deploymentPackageStatusQueueNameResolver.getDeploymentPackageStatusQueueName)
      }
      else {
        session.createTopic(destinationName)
      }
    }

  @Bean
  def jacksonJmsMessageConverter: MessageConverter = {
    val converter: MappingJackson2MessageConverter = new MappingJackson2MessageConverter()
    converter.setTargetType(MessageType.TEXT)
    converter.setTypeIdPropertyName("_type")
    converter.setObjectMapper(ListenerMapper.mapper)
    converter
  }

  private def jmsListenerContainerFactory(defaultJmsErrorHandler: ErrorHandler,
                                          destinationResolver: DestinationResolver
                                         ): JmsListenerContainerFactory[DefaultMessageListenerContainer] = {
    val factory = new DefaultJmsListenerContainerFactory
    factory.setBackOff(new FixedBackOff(taskerSystemProperties.queue.backOff.backoffTimeout, taskerSystemProperties.queue.backOff.backoffAttempts))
    factory.setConnectionFactory(jmsConnectionFactoryProvider.singleConnectionFactory())
    factory.setDestinationResolver(destinationResolver)
    factory.setMessageConverter(jacksonJmsMessageConverter)
    factory.setSessionTransacted(true)
    factory.setErrorHandler(defaultJmsErrorHandler)
    factory
  }
}

trait ArchiveQueueNameResolver {
  def getQueueName: String
}

trait CiQueueNameResolver {
  def getCisChangedQueueName: String
}

trait TaskPathStatusQueueNameResolver {
  def getTaskPathStatusQueueName: String
}

trait DeploymentPackageStatusQueueNameResolver {
  def getDeploymentPackageStatusQueueName: String
}

trait TaskPathStatusQueueFlushResolver {
  def getEventFlushPeriodInMillis: Long
}

trait TaskStepLogQueueFlushResolver {
  def getEventFlushPeriodInMillis: Long
}

trait TaskStepLogQueueNameResolver {
  def getTaskStepLogQueueName: String
}

object ListenerMapper {
  val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)
}
