package com.xebialabs.xldeploy.task.queue

import com.xebialabs.deployit.engine.tasker.TaskReceiver.messages._
import com.xebialabs.deployit.engine.tasker.repository.PendingTaskRepository
import com.xebialabs.deployit.engine.tasker.{TaskId, TaskQueueService}
import grizzled.slf4j.Logging
import javax.jms.Message
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.jms.core.{JmsTemplate, MessagePostProcessor}
import org.springframework.stereotype.Component

@Component
@Autowired
class TaskQueueServiceImpl(@Qualifier("mainJmsTemplate") jmsTemplate: JmsTemplate,
                           pendingTaskRepository: PendingTaskRepository) extends TaskQueueService with Logging {

  @Value("${deploy.task.queue.name:xld-tasks-queue}")
  var taskQueueName: String = _

  override def enqueueTask(taskId: String): Unit = {
    logger.debug(s"About to start enqueueing the task [$taskId]")
    pendingTaskRepository.task(taskId)
      .flatMap(_.workerAddress)
      .map(executeRollbackTask(taskId, _))
      .getOrElse(executeTask(taskId))
  }

  private def executeTask(taskId: TaskId): Unit = {
    logger.debug(s"Sending Execute message for Task [$taskId]")
    jmsTemplate.convertAndSend(taskQueueName, None, postProcessMessage(classOf[ExecuteTask].getName, taskId))
  }

  private def executeRollbackTask(taskId: String, workerAddress: String): Unit = {
    logger.debug(s"Sending ExecuteRollback message for Task [$taskId] to Worker [$workerAddress]")
    jmsTemplate.convertAndSend(workerAddress, None, postProcessMessage(classOf[ExecuteTask].getName, taskId, Some(workerAddress)))
  }

  private def postProcessMessage(msgType: String, taskId: String, workerAddress: Option[String] = None): MessagePostProcessor = (message: Message) => {
    message.setStringProperty("type", msgType)
    message.setStringProperty("taskId", taskId)
    workerAddress.foreach(message.setStringProperty("workerAddress", _))
    message
  }

}
