package com.xebialabs.deployit.test.support;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.*;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;
import org.junit.rules.TemporaryFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.io.Files;
import com.google.common.io.Resources;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.find;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.overthere.util.OverthereUtils.getName;
import static com.xebialabs.platform.test.TestUtils.newInstance;
import static java.lang.String.format;
import static java.util.Arrays.asList;

public class DeployedItestChangeSet {

    public enum TestAction {
        VERIFYABSENCE,
        CREATE,
        VERIFYCREATE,
        MODIFY,
        VERIFYMODIFY,
        DESTROY,
        VERIFYDESTROY,
        NOOP
    }

    public enum PropertyScope {
        DEPLOY,
        INSPECT
    }

    private List<TestAction> tests = newArrayList();
    private TestAction deployMode = TestAction.CREATE;

    @SuppressWarnings("rawtypes")
    private Map<String, Deployed> create = newLinkedHashMap();
    @SuppressWarnings("rawtypes")
    private Map<String, Deployed> modify = newLinkedHashMap();
    @SuppressWarnings("rawtypes")
    private Map<String, Deployed> verifyCreate = newLinkedHashMap();
    @SuppressWarnings("rawtypes")
    private Map<String, Deployed> verifyModify = newLinkedHashMap();
    private List<Type> additionalTypesToDiscover = newArrayList();
    private Set<PropertyDescriptor> requiredForInspection = newHashSet();

    private File changesetXmlFile;
    private Container container;
    private ItestTopology topology;
    private TemporaryFolder folder;

    protected DeployedItestChangeSet(File changesetXmlFile, Container container, ItestTopology topology, TemporaryFolder folder) {
        this.changesetXmlFile = changesetXmlFile;
        this.container = container;
        this.topology = topology;
        this.folder = folder;
    }

    public static DeployedItestChangeSet loadChangeSet(File changesetXmlFile, Container container, ItestTopology topology, TemporaryFolder folder)
            throws JDOMException, IOException {
        DeployedItestChangeSet cs = new DeployedItestChangeSet(changesetXmlFile, container, topology, folder);
        cs.init();
        return cs;
    }

    protected void init() throws JDOMException, IOException {
        SAXBuilder sb = new SAXBuilder(XMLReaders.NONVALIDATING);
        Document deployedProperties = sb.build(changesetXmlFile);

        Element rootElement = deployedProperties.getRootElement();

        setTests(rootElement.getAttributeValue("tests"));
        setDeployMode(rootElement.getAttributeValue("deployMode"));

        String disableInTopology = rootElement.getAttributeValue("disableInTopology");
        if (Strings.isNotBlank(disableInTopology)) {
            Iterable<String> searchTags = Splitter.on(',').trimResults().split(disableInTopology);
            if (!topology.getTags().isEmpty()) {
                for (String searchTag : searchTags) {
                    if (topology.getTags().contains(searchTag)) {
                        // disable the test for this topology
                        return;
                    }
                }
            }
        }

        List<Element> deployedGroupElements = rootElement.getChildren("deployeds");
        for (Element deployedGroupElement : deployedGroupElements) {
            String additionalTypes = deployedGroupElement.getAttributeValue("additionalTypesToDiscover");
            if (Strings.isNotBlank(additionalTypes)) {
                for (String t : additionalTypes.split(",")) {
                    addAdditionalTypeToDiscover(t);
                }
            }

            String target = deployedGroupElement.getAttributeValue("target");
            Container targetContainer = container;

            if (target != null) {
                Type containerType = Type.valueOf(target);
                if (container.getType().instanceOf(containerType)) {
                    // proceed to evaluate deployed specified in this group.
                } else {
                    boolean alwaysDeployToTarget = "true".equals(deployedGroupElement.getAttributeValue("alwaysDeployToTarget"));
                    if (alwaysDeployToTarget) {
                        // This deployed must always be deployed to the specified target as it is a dependency.
                        targetContainer = topology.findFirstMatchingTarget(containerType);
                    } else {
                        // This deployment group is not applicable for the current container. Ignore.
                        continue;
                    }
                }
            }

            List<Element> deployedElements = deployedGroupElement.getChildren();
            File deployedFolder = folder.newFolder();
            for (Element each : deployedElements) {
                convertDeployedElementToDeployed(each, targetContainer, deployedFolder);
            }
        }

        associateDeployedIdsWithContainer();
    }

    protected void setTests(String testsString) {
        if (testsString == null || testsString.trim().isEmpty()) {
            return;
        }

        for (String s : testsString.split("[, ]+")) {
            try {
                tests.add(TestAction.valueOf(s.toUpperCase()));
            } catch (IllegalArgumentException exc) {
                throw new IllegalArgumentException(format("Unsupported tests value [%s] on itest tag, expected one of [%s]", s,
                        on("|").join(newArrayList(TestAction.values()))));
            }
        }
    }

    protected void setDeployMode(String deployModeAttribute) {
        if (deployModeAttribute == null || deployModeAttribute.trim().isEmpty()) {
            return;
        }

        try {
            deployMode = TestAction.valueOf(deployModeAttribute.toUpperCase());
        } catch (IllegalArgumentException exc) {
            throw new IllegalArgumentException(format("Unsupported deployMode value [%s] on itest tag expected [%s]",
                    deployModeAttribute, on("|").join(newArrayList(TestAction.values()))));
        }
    }

    private void convertDeployedElementToDeployed(Element elt, Container container, File deployedFolder) throws IOException {
        Set<String> allowedAttributes = Sets.newHashSet("name", "id", "modification", "replace");
        for (org.jdom2.Attribute a : elt.getAttributes()) {
            if (!allowedAttributes.contains(a.getName())) {
                throw new IllegalArgumentException(String.format("Unknown attribute %s", a.getName()));
            }
        }

        String name = elt.getAttributeValue("name");
        if (name == null) {
            name = checkNotNull(elt.getAttributeValue("id"), "Element [%s] has no [name] attribute", elt.getName());
            logger.warn("Using deprecated 'id' in [{}], instead of 'name' for element [{}].", changesetXmlFile, elt.getName());
        }
        boolean isModification = "true".equals(elt.getAttributeValue("modification"));
        boolean isReplace = "true".equals(elt.getAttributeValue("replace"));

        if (isReplace) {
            checkArgument(deployedToBeCreatedExists(name), "Element [%s] has replace='true' but the original deployed with id [%s] could not be found.", elt.getName(), name);
            addDeployedToModify(createDeployed(elt, container, deployedFolder, name, PropertyScope.DEPLOY));
            addDeployedToVerifyModify(createDeployed(elt, container, deployedFolder, name, PropertyScope.INSPECT));
        } else if (isModification) {
            checkArgument(deployedToBeCreatedExists(name), "Element [%s] has modification='true' but the original deployed with id [%s] could not be found.", elt.getName(), name);

            @SuppressWarnings("unchecked")
            Deployed<?, Container> deployed = cloneDeployed(getDeployedToCreate(name));

            convertPropertyElementsToProperties(elt, deployed, PropertyScope.DEPLOY);

            addDeployedToModify(deployed);
            addDeployedToVerifyModify(deployed);
        } else {
            addDeployedToCreate(createDeployed(elt, container, deployedFolder, name, PropertyScope.DEPLOY));
            addDeployedToVerifyCreate(createDeployed(elt, container, deployedFolder, name, PropertyScope.INSPECT));
        }
    }

    private Deployed<?, Container> createDeployed(Element elt, Container container, File deployedFolder, String name, PropertyScope scope) throws IOException {
        Descriptor d = DescriptorRegistry.getDescriptor(elt.getName());
        checkArgument(d.isAssignableTo(Deployed.class), "Type [%s] is not assignable to udm.Deployed", d.getType());

        Deployed<Deployable, Container> deployed = d.newInstance(name);
        deployed.setContainer(container);

        if (deployed instanceof Artifact) {
            Element fileNameElm = elt.getChild("fileName");
            checkArgument(fileNameElm != null, "Type [%s] is an artifact and must specify fileName that can be resolved from the classpath", d.getType());
            File artifact = new File(deployedFolder, getName(fileNameElm.getValue()));
            URL aURL = getClass().getClassLoader().getResource(fileNameElm.getValue());
            Resources.asByteSource(aURL).copyTo(Files.asByteSink(artifact));
            ((Artifact) deployed).setFile(LocalFile.valueOf(artifact));
        }

        convertPropertyElementsToProperties(elt, deployed, scope);

        Deployable deployable = createDeployableFromDeployed(deployed);
        deployed.setDeployable(deployable);

        return deployed;
    }

    private void convertPropertyElementsToProperties(Element elt, ConfigurationItem deployed, PropertyScope scope) throws IOException {
        Descriptor d = deployed.getType().getDescriptor();

        for (Element p : elt.getChildren()) {
            if (p.getName().equals("fileName")) {
                continue;
            }
            PropertyDescriptor pd = checkNotNull(d.getPropertyDescriptor(p.getName()), "Property [%s] does not exist on type [%s]", p.getName(),
                    deployed.getType());
            if (p.getAttributeValue("inspectionProperty") != null && p.getAttributeValue("inspectionProperty").equals("true")) {
                requiredForInspection.add(pd);
            }

            if (!getScope(p).contains(scope)) {
                continue;
            }

            switch (pd.getKind()) {
                case CI:
                    pd.set(deployed, resolveCiReference(p, pd));
                    break;
                case MAP_STRING_STRING:
                    Map<String, String> map = newHashMap();
                    for (Element e : p.getChildren("entry")) {
                        map.put(Preconditions.checkNotNull(e.getAttributeValue("key")), Preconditions.checkNotNull(e.getValue()));
                    }
                    pd.set(deployed, map);
                    break;
                case SET_OF_STRING:
                    Set<String> setOfString = newHashSet();
                    for (Element v : p.getChildren("value")) {
                        setOfString.add(v.getValue());
                    }
                    pd.set(deployed, setOfString);
                    break;
                case LIST_OF_STRING:
                    List<String> listOfString = newArrayList();
                    for (Element v : p.getChildren("value")) {
                        listOfString.add(v.getValue());
                    }
                    pd.set(deployed, listOfString);
                    break;
                case LIST_OF_CI:
                    List<ConfigurationItem> listOfCi = newArrayList();
                    convertCiCollection(deployed, p, pd, listOfCi, scope);
                    break;
                case SET_OF_CI:
                    Set<ConfigurationItem> setOfCi = newHashSet();
                    convertCiCollection(deployed, p, pd, setOfCi, scope);
                    break;
                default:
                    pd.set(deployed, (p.getAttributeValue("null") != null) ? null : p.getValue());
            }
        }
    }

    protected List<PropertyScope> getScope(Element property) {
        if (property == null) {
            return null;
        }

        String scope = property.getAttributeValue("on");

        if (scope == null || scope.trim().isEmpty()) {
            return asList(PropertyScope.DEPLOY, PropertyScope.INSPECT);
        } else {
            List<PropertyScope> values = new ArrayList<>(2);
            for (String s : scope.split("[, ]+")) {
                try {
                    values.add(PropertyScope.valueOf(s.toUpperCase()));
                } catch (IllegalArgumentException exc) {
                    throw new IllegalArgumentException(format("Unsupported value [%s] on scope tag, expected one of [%s]", s,
                            on("|").join(newArrayList(PropertyScope.values()))));
                }
            }
            return values;
        }
    }

    private void convertCiCollection(ConfigurationItem deployed, Element p, PropertyDescriptor pd, Collection<ConfigurationItem> cis, PropertyScope scope) throws IOException {
        if (pd.isAsContainment()) {
            List<Element> embeddeds = p.getChildren();
            for (Element ee : embeddeds) {
                String name = checkNotNull(ee.getAttributeValue("name"), "Element [%s] has no [name] attribute", ee.getName());
                Descriptor ed = DescriptorRegistry.getDescriptor(ee.getName());
                checkArgument(ed.isAssignableTo(EmbeddedDeployed.class), "Type [%s] is not assignable to udm.EmbeddedDeployed", ed.getType());
                EmbeddedDeployed<EmbeddedDeployable, EmbeddedDeployedContainer<?, ?>> embeddedDeployed = ed.newInstance(deployed.getId() + "/" + name);
                convertPropertyElementsToProperties(ee, embeddedDeployed, scope);
                embeddedDeployed.setContainer((EmbeddedDeployedContainer<?, ?>) deployed);
                cis.add(embeddedDeployed);
            }
        } else {
            for (Element v : p.getChildren("value")) {
                cis.add(resolveCiReference(v, pd));
            }
        }
        pd.set(deployed, cis);
    }

    protected ConfigurationItem resolveCiReference(Element p, PropertyDescriptor pd) {
        String ciRefId = p.getValue().trim();
        if (deployedToBeCreatedExists(ciRefId)) {
            return getDeployedToCreate(ciRefId);
        }

        ciRefId = topology.replacePlaceholders(ciRefId);
        ConfigurationItem ciRef = topology.getItems().get(ciRefId);
        if (ciRef == null) {
            ciRef = DescriptorRegistry.getDescriptor(pd.getReferencedType()).newInstance(ciRefId);
        }

        return ciRef;
    }

    private Deployable createDeployableFromDeployed(Deployed<?, Container> deployed) {
        Descriptor deployedDesc = deployed.getType().getDescriptor();
        Descriptor deployableDesc = deployedDesc.getDeployableType().getDescriptor();

        // if the defined deployable is a virtual, find the first applicable subtype which can be instantiated
        if (deployableDesc.isVirtual()) {
            Type instantiableDeployable = find(DescriptorRegistry.getSubtypes(deployableDesc.getType()), new Predicate<Type>() {
                @Override
                public boolean apply(Type subType) {
                    return !subType.getDescriptor().isVirtual();
                }
            }, null);
            checkNotNull(instantiableDeployable, "Unable to find instantiable subtype of virtual type %s. Please check your CI definitions.", deployableDesc.getType().getName());
            deployableDesc = instantiableDeployable.getDescriptor();
        }

        return deployableDesc.newInstance(deployed.getId());
    }

    @SuppressWarnings("rawtypes")
    protected void associateDeployedIdsWithContainer() {
        Iterable<Deployed> deployeds = Iterables.concat(getDeployedsToCreate(), getDeployedsToModify(), getDeployedsToVerifyCreate(), getDeployedsToVerifyModify());
        for (Deployed deployed : deployeds) {
            Preconditions.checkNotNull(deployed.getContainer());
            deployed.setId(deployed.getContainer().getId() + "/" + deployed.getId());
        }
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    protected List<Deployed> cloneForInspection(List<Deployed> deployeds) {
        List<Deployed> clonedDeployeds = newArrayList();
        for (Deployed<?, ?> deployed : deployeds) {
            Descriptor descriptor = DescriptorRegistry.getDescriptor(deployed.getType());
            Deployed c = descriptor.newInstance(deployed.getId());
            Iterable<PropertyDescriptor> inspectProperties = filter(descriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
                @Override
                public boolean apply(PropertyDescriptor input) {
                    return input.isInspectionProperty() || requiredForInspection.contains(input);
                }
            });
            c.setDeployable(deployed.getDeployable());
            c.setContainer(deployed.getContainer());
            for (PropertyDescriptor inspectProperty : inspectProperties) {
                inspectProperty.set(c, inspectProperty.get(deployed));
            }
            clonedDeployeds.add(c);
        }
        return clonedDeployeds;

    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    protected Deployed cloneDeployed(Deployed<?, Container> deployed) {
        Deployed clonedDeployed = newInstance(deployed.getType(), deployed.getId());
        clonedDeployed.setDeployable(deployed.getDeployable());
        clonedDeployed.setContainer(deployed.getContainer());
        for (PropertyDescriptor inspectProperty : deployed.getType().getDescriptor().getPropertyDescriptors()) {
            inspectProperty.set(clonedDeployed, inspectProperty.get(deployed));
        }

        try {
            Method getFile = deployed.getClass().getMethod("getFile");
            Method setFile = clonedDeployed.getClass().getMethod("setFile", OverthereFile.class);
            setFile.invoke(clonedDeployed, getFile.invoke(deployed));
        } catch (Exception e) {
            // fine, don't set file
        }

        return clonedDeployed;
    }

    public List<TestAction> getTests() {
        return tests;
    }

    public TestAction getDeployMode() {
        return deployMode;
    }

    public List<Type> getAdditionalTypesToDiscover() {
        return additionalTypesToDiscover;
    }

    public boolean hasDeployedsToCreate() {
        return !create.isEmpty();
    }

    public boolean hasDeployedsToModify() {
        return !modify.isEmpty();
    }

    public void addAdditionalTypeToDiscover(String type) {
        additionalTypesToDiscover.add(Type.valueOf(type.trim()));
    }

    @SuppressWarnings("rawtypes")
    public void addDeployedToCreate(Deployed d) {
        create.put(d.getId(), d);
    }

    @SuppressWarnings("rawtypes")
    public void addDeployedToVerifyCreate(Deployed d) {
        verifyCreate.put(d.getId(), d);
    }

    @SuppressWarnings("rawtypes")
    public void addDeployedToModify(Deployed d) {
        modify.put(d.getId(), d);
    }

    @SuppressWarnings("rawtypes")
    public void addDeployedToVerifyModify(Deployed d) {
        verifyModify.put(d.getId(), d);
    }

    @SuppressWarnings("rawtypes")
    public Deployed getDeployedToCreate(String id) {
        return create.get(id);
    }

    @SuppressWarnings("rawtypes")
    public Deployed getDeployedToModify(String id) {
        return modify.get(id);
    }

    public boolean deployedToBeCreatedExists(String id) {
        return create.containsKey(id);
    }

    @SuppressWarnings("rawtypes")
    public List<Deployed> getDeployedsToCreate() {
        return newArrayList(create.values());
    }

    @SuppressWarnings("rawtypes")
    public List<Deployed> getDeployedsToVerifyCreate() {
        return newArrayList(verifyCreate.values());
    }

    @SuppressWarnings("rawtypes")
    public List<Deployed> getDeployedsToVerifyModify() {
        return newArrayList(verifyModify.values());
    }

    @SuppressWarnings("rawtypes")
    public List<Deployed> getDeployedsToModify() {
        return newArrayList(modify.values());
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedItestChangeSet.class);
}
