package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, Operation}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.{Scope, StepPostConstructContext}
import com.xebialabs.deployit.script.SpringBindings
import com.xebialabs.platform.script.jython.{ScriptSource}
import com.xebialabs.platform.script.jython.DeployJythonSupport
import com.xebialabs.xlplatform._
import com.xebialabs.xlplatform.script.jython.JythonSugarDiscovery
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

class ScriptPlanRule(name: String,
                     scope: Scope,
                     scriptSource: ScriptSource,
                     override val condition: Option[String] = None)
  extends ScriptRule(name, scope, scriptSource, condition) with PlanRule

class ScriptDeployedRule(name: String,
                         scriptSource: ScriptSource,
                         override val ciTypes: Iterable[Type],
                         override val operations: Iterable[Operation],
                         override val condition: Option[String] = None)
  extends ScriptRule(name, Scope.DEPLOYED, scriptSource, condition) with DeployedRule

abstract class ScriptRule(name: String, scope: Scope, scriptSource: ScriptSource, condition: Option[String]) extends Rule(name, scope) with ContextFactory with Logging with DeployJythonSupport {

  override def doFire(scopedObject: AnyRef, context: RulePlanningContext): Unit = {
    implicit val pContext: RulePlanningContext = context
    executeScript(scriptSource)(jythonContext(scopedObject, getScope))
  }

  def getScript: String = scriptSource.scriptContent
}

class ScriptStepFactory(stepRegistry: StepRegistry, stepPostConstructContext: StepPostConstructContext)
  extends StepFactory(stepRegistry, stepPostConstructContext, new XmlParameterResolver(stepRegistry)) {

  def stepFromScript(name: String, args: java.util.List[Any], namedParameters: java.util.Map[String, Any], existingBindings: Map[String, Any] = Map()): Step = {
    validateArguments(args.asScala.toList)
    step(StepData(name.underscoresToDashes, namedParameters.asScala.toMap.map { entry => (entry._1.underscoresToDashes, entry._2) }), existingBindings)
  }

  private def validateArguments(args: List[Any]): Unit = {
    if (args != null && args.nonEmpty) {
      throw new IllegalArgumentException("Only named parameters are supported")
    }
  }
}

trait ContextFactory extends Logging {

  import com.xebialabs.platform.script.jython._

  def stepPostConstructContext(scope: Scope, scopedObject: AnyRef)(implicit context: RulePlanningContext) =
    new StepPostConstructContext(scope, scopedObject, context.getDeployedApplication, context.getRepository)

  def jythonContext(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext): JythonContext = {
    val bindings = allBindings(scopedObject, scope)
    JythonContext(
      (Syntactic.wrapperCodeWithLib(bindings.keySet) :+ ruleLib :+ Syntactic.loggerLib) ++ JythonSugarDiscovery.getExtensionResources,
      bindings
    )
  }

  def jythonContext(bindings: Map[String, Any]): JythonContext = {
    JythonContext(
      (Syntactic.wrapperCodeWithLib(bindings.keySet) :+ ruleLib :+ Syntactic.loggerLib) ++ JythonSugarDiscovery.getExtensionResources,
      bindings
    )
  }

  def allBindings(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext): Map[String, Object] = {
    val bindings: Map[String, Object] = scopedBindings(scopedObject, scope) ++ unscopedBindings(createStepFactory(scopedObject, scope)) ++ Bindings.xlDeployApiServices ++ SpringBindings.bindings.get()
    bindings ++ Map("_bindings" -> bindings)
  }

  private def scopedBindings(scopedObject: AnyRef, scope: Scope): Map[String, AnyRef] = scope match {
    case Scope.DEPLOYED => Map(
      ContextFactory.DELTA -> scopedObject,
      ContextFactory.DEPLOYED -> scopedObject.asInstanceOf[Delta].getDeployed,
      ContextFactory.PREVIOUS_DEPLOYED -> scopedObject.asInstanceOf[Delta].getPrevious
    )
    case Scope.PLAN => Map(ContextFactory.DELTAS -> scopedObject)
    case Scope.PRE_PLAN | Scope.POST_PLAN => Map(ContextFactory.SPECIFICATION -> scopedObject)
    case _ => Map()
  }

  private def unscopedBindings(stepFactory: ScriptStepFactory)(implicit context: RulePlanningContext) = Map(
    ContextFactory.DEPLOYED_APPLICATION -> context.getDeployedApplication,
    ContextFactory.PREVIOUS_DEPLOYED_APPLICATION -> context.getPreviousDeployedApplication,
    ContextFactory.CONTEXT -> context,
    ContextFactory.INTERNAL_STEP_FACTORY -> stepFactory,
    ContextFactory.GLOBAL_CONTEXT -> context.getGlobalDeploymentPlanningContext()
  )

  private def createStepFactory(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext): ScriptStepFactory = {
    new ScriptStepFactory(context.getStepRegistry, stepPostConstructContext(scope, scopedObject))
  }
}

object ContextFactory {
  val CONTEXT = "context"
  val GLOBAL_CONTEXT = "globalContext"
  val DELTA = "delta"
  val DELTAS = "deltas"
  val DEPLOYED = "deployed"
  val DEPLOYED_APPLICATION = "deployedApplication"
  val INTERNAL_STEP_FACTORY = "_internal_step_factory"
  val PREVIOUS_DEPLOYED = "previousDeployed"
  val PREVIOUS_DEPLOYED_APPLICATION = "previousDeployedApplication"
  val SPECIFICATION = "specification"
}