package com.xebialabs.deployit.deployment.rules


import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Deltas}
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

trait RulePlannerHelper extends Logging {
  def addStepsOfScopeDeployed(deltas: Seq[Delta],
                              stepPlan: StepPlan,
                              planningContext: RulePlanningContext,
                              ruleRegistry: RuleRegistry): Unit = {
    val deployedRules = ruleRegistry.getEnabledRules(Scope.DEPLOYED)
    deltas.foreach { delta =>
      try {
        stepPlan.setDeltaUnderPlanning(delta)
        deployedRules.foreach(fireAndLog(_, stepPlan, delta, planningContext))
      } finally {
        stepPlan.setDeltaUnderPlanning(null)
      }
    }
  }

  def stepPlanOfScopePrePostPlan(spec: DeltaSpecification, planExecutionContext: PlanCreationContext, scope: Scope): StepPlan = {
    val plan = planExecutionContext.stepPlan("Partial pre/post plan")
    val ctx = planExecutionContext.planningContext(plan, spec)
    planExecutionContext.ruleRegistry.getEnabledRules(scope).foreach(fireAndLog(_, plan, spec, ctx))
    plan
  }

  def addStepsOfPlanScope(deltas: Seq[Delta],
                          stepPlan: StepPlan,
                          planningContext: RulePlanningContext,
                          ruleRegistry: RuleRegistry): Unit =
    ruleRegistry.getEnabledRules(Scope.PLAN).foreach(fireAndLog(_, stepPlan, new Deltas(deltas.asJava), planningContext))

  def fireAndLog(rule: Rule, stepPlan: StepPlan, scopedObject: AnyRef, ctx: RulePlanningContext): Unit = {
    def deploymentMessage(application: Option[DeployedApplication]) = {
      s"${application.map(_.getVersion).orNull} to ${application.map(_.getEnvironment).orNull}"
    }

    try {
      stepPlan.setRuleUnderPlanning(rule.getName)
      trace(s"Checking conditions for rule $rule")
      if (rule.canFire(scopedObject, ctx)) {
        debug(
          s"""Firing rule ${rule.getName} while
              |deployed application: ${deploymentMessage(Option(ctx.getDeployedApplication))}
              |and previously deployed application: ${deploymentMessage(Option(ctx.getPreviousDeployedApplication))}""".stripMargin)
        rule.doFire(scopedObject, ctx)
      }
    } catch {
      case ex: Exception =>
        error(s"Error while executing rule $rule for scopedObject: $scopedObject", ex)
        throw PlannerException(s"Error while evaluating rule [${rule.getName}]: ${ex.getMessage}", ex)
    } finally {
      stepPlan.setRuleUnderPlanning(null)
    }
  }
}
