package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PipedPlanner.{PlanProducer, PlanTransformer, TransformerContext}
import com.xebialabs.deployit.deployment.rules.PlanCreationContext
import grizzled.slf4j.Logging
import org.springframework.util.StopWatch

/**
 * A variation of {@see Planner} that measures plan production and transformation time and prints it to the debug log
 * at the end of planning operation
 *
 * This class uses {@see PipedPlanner} inside to perform planning
 * @param producer - {@see PlanProducer} instance
 * @param transformers sequence of {@see PlanTransformer} instances
 */
class PerformanceAwarePipedPlanner(producer: PlanProducer, transformers: Seq[PlanTransformer]) extends Planner with Logging {
  private def measure[T](pipeElement: T, call: T => PhasedPlan)(implicit watch: StopWatch): PhasedPlan = {
    watch.start(pipeElement.getClass.getCanonicalName)
    val plan = call(pipeElement)
    watch.stop()
    plan
  }

  private def prepareProducer(implicit watch: StopWatch): PlanProducer = (spec: MultiDeltaSpecification,
                                                                          planCreationContext: PlanCreationContext) => {
    info(s"Creating initial plan using: ${producer.getClass.getCanonicalName}")
    measure[PlanProducer](producer, _.produce(spec, planCreationContext))
  }

  private def prepareTransformers(implicit watch: StopWatch): Seq[PlanTransformer] = transformers.map { transformer =>
    new PlanTransformer {
      override def transform(context: TransformerContext): PhasedPlan = {
        info(s"Applying plan transformer: ${transformer.getClass.getCanonicalName}")
        measure[PlanTransformer](transformer, _.transform(context))
      }
    }
  }

  override def plan(spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext): PhasedPlan = {
    implicit val watch: StopWatch = new StopWatch()
    val plan = new PipedPlanner(prepareProducer, prepareTransformers).plan(spec, planCreationContext)
    info(s"\n${watch.prettyPrint()}")
    plan
  }
}
