package com.xebialabs.deployit.deployment.orchestrator

import java.util

import com.xebialabs.deployit.deployment.orchestrator.OrchestratorComposer.FilteredDeltaSpecificationWithDependencies
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrations._
import com.xebialabs.deployit.engine.spi.orchestration._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, DeltaSpecificationWithDependencies, Operation}
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.xlplatform.utils.Implicits._

import scala.jdk.CollectionConverters._

object OrchestratorComposer {

  private class FilteredDeltaSpecificationWithDependencies(deltasInScope: List[Delta],
                                                           wrappedSpec: DeltaSpecificationWithDependencies) extends DeltaSpecificationWithDependencies {
    override def getAllDeltaSpecifications: util.List[DeltaSpecification] =
      filterSpecs(deltasInScope, wrappedSpec.getAllDeltaSpecifications.asScala.toList).asJava

    override def getAllGroupedDeltaSpecifications: util.List[util.List[DeltaSpecification]] =
      filterGroupedSpecs(deltasInScope, wrappedSpec.getAllGroupedDeltaSpecifications)

    private def filterGroupedSpecs(deltasInScope: List[Delta], specs: List[List[DeltaSpecification]]) = {
      val mainSpec: DeltaSpecification = specs.lastOption.flatMap(_.lastOption).getOrElse(throw new NoSuchElementException("Cannot get main spec"))
      val deltaSet = deltasInScope.toSet
      specs.map(_.filter(x => x.getDeltas.asScala.exists(deltaSet))).map(filterSpecs(deltasInScope, _, mainSpec))
    }

    private def filterSpecs(deltasInScope: List[Delta], specs: List[DeltaSpecification]): List[DeltaSpecification] = {
      specs match {
        case head :: Nil => new FilteredDeltaSpecification(deltasInScope, head) :: Nil
        case list if list.nonEmpty => filterSpecs(deltasInScope, list, list.lastOption.orNull)
      }
    }

    private[this] def filterSpecs(deltasInScope: List[Delta], specs: List[DeltaSpecification], mainSpec: DeltaSpecification): List[DeltaSpecification] = {
      deltasInScope.groupBy(d => specs.find(_.getDeltas.contains(d)).getOrElse(mainSpec)).map { case (s, d) => new FilteredDeltaSpecification(d, s)}.toList
    }
  }


  private class FilteredDeltaSpecification(theDeltas: List[Delta], wrappedSpec: DeltaSpecification) extends DeltaSpecification {
    override def getDeployedApplication: DeployedApplication = wrappedSpec.getDeployedApplication

    override def getDeltas: util.List[Delta] = theDeltas.asJava

    override def getOperation: Operation = wrappedSpec.getOperation

    override def getPreviousDeployedApplication: DeployedApplication = wrappedSpec.getPreviousDeployedApplication

    override def isRollback: Boolean = wrappedSpec.isRollback
  }
}

class OrchestratorComposer {

  def orchestrate(orchestartors: List[Orchestrator], spec: DeltaSpecificationWithDependencies): Orchestration = {
    orchestartors match {
      case Nil =>
        // TODO DEPL-9919 Add sequential-by-application orchestrator here if deps are detected ?
        new DefaultOrchestrator().orchestrate(spec)
      case head :: tail =>
        tail.foldLeft(head.orchestrate(spec)) { case (previousResult, orchestrator) =>
          applyOrchestrator(previousResult, orchestrator, spec)
        }
    }
  }

  private def applyOrchestrator(orchestration: Orchestration, orchestrator: Orchestrator, spec: DeltaSpecificationWithDependencies): Orchestration = orchestration match {
    case s: SerialOrchestration =>
      serial(s.getDescription, s.getPlans.asScala.map(applyOrchestrator(_, orchestrator, spec)).asJava)
    case p: ParallelOrchestration =>
      parallel(p.getDescription, p.getPlans.asScala.map(applyOrchestrator(_, orchestrator, spec)).asJava)
    case i: InterleavedOrchestration if !i.getDeltas.isEmpty =>
      orchestrator.orchestrate(new FilteredDeltaSpecificationWithDependencies(i.getDeltas.asScala.toList, spec)) match {
        case s: SerialOrchestration => serial(i.getDescription, s.getPlans)
        case p: ParallelOrchestration => parallel(i.getDescription, p.getPlans)
        case i2: InterleavedOrchestration => interleaved(i.getDescription, i2.getDeltas)
      }
    case i: InterleavedOrchestration => i
  }
}
