package com.xebialabs.deployit.deployment.rules

import java.lang.reflect.Field
import java.util

import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem

import scala.reflect.ClassTag
object StepPrimitiveParameterDescriptor {

  implicit class Utils(descriptor: StepPrimitiveParameterDescriptor) {
    def isFieldOfType[T: ClassTag] = implicitly[ClassTag[T]].runtimeClass.isAssignableFrom(descriptor.field.getType)
  }

}

sealed trait StepParameterDescriptor {
  val name: String
  val description: String
  val required: Boolean
  val parameterType: ParameterType
}

object StepParameterDescriptor {

  implicit class Utils(descriptor: StepParameterDescriptor) {
    def isFieldOfType[T: ClassTag] = implicitly[ClassTag[T]].runtimeClass.isAssignableFrom(descriptor.parameterType.fieldType)
  }

}


case class StepPrimitiveParameterDescriptor(description: String, field: Field, required: Boolean) extends StepParameterDescriptor {
  override val name: String = field.getName

  @SuppressWarnings(Array("ComparingUnrelatedTypes"))
  override val parameterType: ParameterType = field.getType match {
    case t if classOf[String].isAssignableFrom(t) => StringParameterType
    case t if classOf[Integer].isAssignableFrom(t) || t == Integer.TYPE => IntegerParameterType
    case t if classOf[util.Map[String, _]].isAssignableFrom(t) => MapParameterType
    case t if classOf[util.List[String]].isAssignableFrom(t) => ListParameterType
    case t if classOf[util.Set[String]].isAssignableFrom(t) => SetParameterType
    case t if classOf[ConfigurationItem].isAssignableFrom(t) => CiParameterType
    case t if classOf[java.lang.Boolean].isAssignableFrom(t) || t == java.lang.Boolean.TYPE => BooleanParameterType
    case _ => throw new IllegalArgumentException(s"Unable to resolve step parameter with name ${field.getName} and type ${field.getType}")
  }


}

case class StepMacroParameterDescriptor(name: String, description: String, parameterType: ParameterType, required: Boolean = false) extends StepParameterDescriptor


sealed trait StepDescriptor {
  val name: String

  def parameters: Map[String, StepParameterDescriptor]

}

case class StepPrimitiveDescriptor(name: String, implementationClass: Class[_ <: Step], parameters: Map[String, StepPrimitiveParameterDescriptor]) extends StepDescriptor

case class StepMacroDescriptor(name: String, stepData: XmlStepData, parameters: Map[String, StepMacroParameterDescriptor]) extends StepDescriptor


trait ParameterType {
  def fieldType: Class[_]
}

case object StringParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[String]
}

case object IntegerParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[Integer]
}

case object BooleanParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[Boolean]
}

case object ListParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[util.List[_]]
}

case object SetParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[util.Set[_]]
}

case object MapParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[util.Map[_, _]]
}

case object CiParameterType extends ParameterType {
  override def fieldType: Class[_] = classOf[ConfigurationItem]
}
