package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.deployment.rules.RuleParser._
import grizzled.slf4j.Logging

import scala.xml.{Elem, Node}

private[rules] class XmlStepMacroParser(val stepRegistry: StepRegistry) extends Logging with StepParser with StepMacroParameterParser {

  def stepMacrosFromXml(root: Elem) = {
    trace("Finding all step-macro elements")
    val stepMacroDescriptors = root \ "step-macro" map parse
    stepMacroDescriptors.foreach(descriptor => stepRegistry.register(descriptor))
  }

  private[this] def parse(stepMacroNode: Node): StepMacroDescriptor = {
    val macroName = (stepMacroNode \ "@name").text
    val stepMacros = stepMacroNode \ "steps" flatMap (_.notTextChild) collect step(stepRegistry)
    stepMacros match {
      case (head: XmlStepData) :: Nil => StepMacroDescriptor(macroName, head, parameters(stepMacroNode))
      case head :: tail => throw new IllegalArgumentException("Step macro currently does not support more than one step")
      case Nil => throw new IllegalArgumentException("Step macro need one step to work")
    }
  }

}

private[rules] trait StepMacroParameterParser {

  def parameters(stepMacroNode: Node): Map[String, StepMacroParameterDescriptor] = {
    val parametersNode: Option[Node] = (stepMacroNode \ "parameters").headOption
    parametersNode.map(ps => ps \ "parameter").map(ps => ps.map(p => toXmlStepMacroParameter(p)).map(p => p.name -> p).toMap).getOrElse(Map())
  }

  private[this] def toXmlStepMacroParameter(parameter: Node): StepMacroParameterDescriptor = {
    val name: String = attr(parameter, "name")
    val description: String = attr(parameter, "description")
    val pt: ParameterType = parameterType(attr(parameter, "type"))
    StepMacroParameterDescriptor(name, description, pt)
  }

  private[this] def attr(node: Node, attr: String) = (node \ s"@$attr").text


  private[this] def parameterType(parameterType: String): ParameterType = {
    parameterType match {
      case "string" => StringParameterType
      case "integer" => IntegerParameterType
      case "boolean" => BooleanParameterType
      case "ci" => CiParameterType
      case "set_of_string" => SetParameterType
      case "list_of_string" => ListParameterType
      case "map_string_string" => MapParameterType
      case _ => throw new IllegalArgumentException(s"Unsupported parameter type $parameterType. Only ['string','integer','boolean','ci','set_of_string','list_of_string','map_string_string'] are supported.")
    }
  }
}
