package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.deployment.rules.RegistriesActor.Messages._
import com.xebialabs.xlplatform.sugar.ResourcesSugar
import com.xebialabs.xlplatform.watcher.FileWatchActor.Messages._
import grizzled.slf4j.Logging
import nl.javadude.scannit.Scannit
import org.apache.pekko.actor._

import java.io.File
import java.nio.file.Path

private[rules] object RegistriesActor {
  def props(fileWatchActor: ActorRef): Props = Props(new RegistriesActor(fileWatchActor))

  object Messages {

    case class RuleRegistryGet()

    case class StepRegistryGet()

    case class StepMacroRegistryGet()

    case class RuleRegistryReturn(ruleRegistry: RuleRegistry)

    case class StepRegistryReturn(stepRegistry: StepRegistry)

  }

}

private[rules] class RegistriesActor(fileWatchActor: ActorRef) extends Actor with Registries with ResourceScanningSubscription with ActorLogging {

  var steps: StepRegistry = _
  var rules: RuleRegistry = _

  override def preStart(): Unit = {
    subscribeForFileChanges()
    initStepsAndRules()
  }

  override def receive: Receive = {
    case RuleRegistryGet() => sender() ! RuleRegistryReturn(rules)
    case StepRegistryGet() => sender() ! StepRegistryReturn(steps)
    case Created(_) | Modified(_) | Deleted(_) => initStepsAndRules()
    case WatchStopped(path) => subscribeForFileChange(path)
    case StartWatching(path) => log.debug(s"Watching on $path")
  }

  def initStepsAndRules(): Unit = {
    steps = stepRegistry()
    addStepMacros(steps)
    rules = ruleRegistry(steps)
  }

  override def subscribeForFileChange(path: Path): Unit = fileWatchActor ! Watch(path)
}

private[deployit] trait Registries extends Logging {

  def ruleRegistry(stepRegistry: StepRegistry): SimpleRuleRegistry = {
    try {
      val ruleRegistry = new SimpleRuleRegistry()
      logger.info("Registering deployment rules...")
      JavaBasedRuleBuilder.fill(ruleRegistry)
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlRuleParser(ruleRegistry, stepRegistry).ruleFromXml)
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlRuleParser(ruleRegistry, stepRegistry).disableRulesXml)
      ruleRegistry.preloadScriptRuleSources()
      logger.info("Deployment rule registration: done.")
      ruleRegistry
    } catch {
      case ex: Exception =>
        error("Error while creating rule registry", ex)
        throw ex
    }
  }

  def stepRegistry(scannit: Scannit = Scannit.getInstance()): SimpleStepRegistry = {
    val stepRegistry = new SimpleStepRegistry()
    StepClassScanner.fill(stepRegistry, scannit)
    stepRegistry
  }

  def addStepMacros(stepRegistry: StepRegistry): StepRegistry = {
    try {
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlStepMacroParser(stepRegistry).stepMacrosFromXml)
      stepRegistry
    }
    catch {
      case ex: Exception =>
        error("Error while create step macro registry", ex)
        throw ex
    }
  }

}

private[rules] trait ResourceScanningSubscription extends ResourcesSugar {

  def subscribeForFileChanges(): Unit =
    resources(rulesXmlFileName).filter(_.getProtocol == "file").foreach { resource =>
      subscribeForFileChange(new File(resource.toURI).toPath)
    }

  def subscribeForFileChange(path: Path): Unit
}
