package com.xebialabs.deployit.deployment.orchestrator;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.xebialabs.deployit.deployment.planner.MultiDeltaSpecificationHelper;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecificationWithDependencies;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.*;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.deployment.planner.MultiDeltaSpecificationHelper.commonOperation;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.*;

public final class DescriptionHelper {

    private DescriptionHelper() {}

    private static Map<Operation, String> deploymentVerbs = ImmutableMap.of(
            CREATE, "Deploy",
            DESTROY, "Undeploy",
            MODIFY, "Update",
            NOOP, "Not update");

    private static Map<Operation, String> provisioningVerbs = ImmutableMap.of(
            CREATE, "Provision",
            DESTROY, "Deprovision",
            MODIFY, "Update",
            NOOP, "Not update");

    private static Joiner joinWithSpaces = Joiner.on(" ");

    public static String getDescriptionForSpecWithDependencies(DeltaSpecificationWithDependencies specs) {
        return getDescriptionForSpecs(specs.getAllDeltaSpecifications());
    }

    public static String getDescriptionForSpecs(List<DeltaSpecification> allDeltaSpecifications) {
        String appAndVersionNames = allDeltaSpecifications.stream()
                .map(MultiDeltaSpecificationHelper::correctDeployedApplication)
                .map(da -> da.getName() + " " + da.getVersion().getVersion())
                .collect(Collectors.joining(", "));
        String environmentName = allDeltaSpecifications.stream()
                .map(MultiDeltaSpecificationHelper::correctDeployedApplication)
                .findAny()
                .map(DeployedApplication::getEnvironment)
                .map(Environment::getName)
                .orElse("No Environment");

        String verb = allDeltaSpecifications.stream()
                .map(MultiDeltaSpecificationHelper::correctDeployedApplication)
                .findFirst()
                .map(DeployedApplication::getVersion)
                .filter(version -> version instanceof ProvisioningPackage)
                .map(pp -> provisioningVerbs.get(commonOperation(allDeltaSpecifications).orElse(CREATE)))
                .orElseGet(() -> deploymentVerbs.get(commonOperation(allDeltaSpecifications).orElse(CREATE)));

        return joinWithSpaces.join(
                verb,
                appAndVersionNames,
                "on",
                environmentName);
    }


    public static String getDescriptionForSpec(DeltaSpecification specification) {
        DeployedApplication deployedApplication = specification.getDeployedApplication();
        return joinWithSpaces.join(
                verb(specification, deployedApplication),
                deployedApplication.getName(),
                deployedApplication.getVersion().getName(),
                "on",
                deployedApplication.getEnvironment().getName());
    }

    private static String verb(DeltaSpecification specification, DeployedApplication deployedApplication) {
        if (deployedApplication.getVersion() instanceof ProvisioningPackage) {
            return provisioningVerbs.get(specification.getOperation());
        }
        return deploymentVerbs.get(specification.getOperation());
    }

    public static String getDescriptionForContainer(Operation op, Container con) {
        return joinWithSpaces.join(
                deploymentVerbs.get(op),
                "on",
                con.getName());
    }

    public static String getDescriptionForPackage(Operation op, Version version) {
        return joinWithSpaces.join(
                deploymentVerbs.get(op),
                "package",
                version.getApplication().getName(),
                version.getName());
    }

}
