package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PipedPlanner.{PlanProducer, PlanTransformer, TransformerContext}
import com.xebialabs.deployit.deployment.rules.PlanCreationContext
import grizzled.slf4j.Logging

object PipedPlanner {
  trait PlanProducer {
    def produce(spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext): PhasedPlan
  }

  case class TransformerContext(plan: PhasedPlan, spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext)

  /**
   * The implementation of this interface should transform the plan passed to it and return a new instance of the plan
   */
  trait PlanTransformer {
    /**
     * Transforms the plan passed to it inside of {@see TransformerContext} and returns a new instance of the plan
     * (transformed plan). Please note: this function should not modify an existing plan, but rather create a new one
     * based on an old plan
     * @param context - plan transformation context that hold an old plan and some planning - related instances that can
     *                be used during transformation
     * @return new (transformed) plan
     */
    def transform(context: TransformerContext): PhasedPlan
  }

  /**
   * Creates a "performance aware" instance of PipedPlanner. It will catch the time that each operation takes and prints
   * it to debug log at the end of planning
   * @param producer - {@see PlanProducer} instance
   * @param transformers sequence of {@see PlanTransformer} instances
   * @return new "performance aware" {@see PipedPlanner} instance
   */
  def performanceAware(producer: PlanProducer, transformers: Seq[PlanTransformer]): Planner =
    new PerformanceAwarePipedPlanner(producer, transformers)
}

/**
 * Implementation of {@see Planner} interface that use {@see PlanProducer} instance to build initial plan and sequentially
 * passes it through pipe of plan transformers {@see PlanTransformer} before returning it to the user
 * @param producer - {@see Planner} instance that builds the initial plan
 * @param transformers - sequence of {@see PlanTransformer} instances that transforms plan
 */
class PipedPlanner(producer: PlanProducer, transformers: Seq[PlanTransformer]) extends Planner with Logging {
  override def plan(spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext): PhasedPlan = {
    val plan = producer.produce(spec, planCreationContext)
    val initial = TransformerContext(plan, spec, planCreationContext)
    transformers.foldLeft(initial) { case (context, transformer) =>
      val transformedPlan = transformer.transform(context)
      context.copy(plan = transformedPlan)
    }.plan
  }
}
